<?php

use App\Http\Controllers\Api\Auth\AuthController;
use App\Http\Controllers\Api\Customer\MenuController as CustomerMenuController;
use App\Http\Controllers\Api\Customer\CartController as CustomerCartController;
use App\Http\Controllers\Api\Customer\OrderController as CustomerOrderController;
use App\Http\Controllers\Api\Customer\PaymentController as CustomerPaymentController;
use App\Http\Controllers\Api\Customer\ReviewController as CustomerReviewController;
use App\Http\Controllers\Api\Customer\CareerController as CustomerCareerController;

use App\Http\Controllers\Api\Auth\AuthController as EmployeeAuthController;
use App\Http\Controllers\Api\Employee\OrderController as EmployeeOrderController;
use App\Http\Controllers\Api\Employee\PaymentConfirmationController as EmployeePaymentController;
use Illuminate\Support\Facades\Route;

// Public auth routes
Route::prefix('auth')->group(function () {
    Route::post('/register', [AuthController::class, 'register']);
    Route::post('/login', [AuthController::class, 'login']);

    // Health check
    Route::get('/health', function () {
        return response()->json([
            'success' => true,
            'message' => 'Chingu Bite API is running',
            'timestamp' => now()->toDateTimeString(),
            'version' => '1.0.0'
        ]);
    });
});

Route::prefix('customer')->group(function () {
    // Menu routes
    Route::prefix('menu')->group(function () {
        Route::get('/categories', [CustomerMenuController::class, 'getCategories']);
        Route::get('/', [CustomerMenuController::class, 'index']);
        Route::get('/special', [CustomerMenuController::class, 'getSpecialMenus']);
        Route::get('/best-seller', [CustomerMenuController::class, 'getBestSellers']);
        Route::get('/{id}', [CustomerMenuController::class, 'show']);
    });

    // Promo routes
    Route::prefix('promos')->group(function () {
        Route::get('/', [CustomerMenuController::class, 'getPromos']);
        Route::post('/check', [CustomerMenuController::class, 'checkPromo']);
    });
});

// Protected routes (auth required)
Route::middleware('auth:sanctum')->group(function () {
    // Auth routes
    Route::prefix('auth')->group(function () {
        Route::get('/me', [AuthController::class, 'me']);
        Route::put('/profile', [AuthController::class, 'updateProfile']);
        Route::put('/change-password', [AuthController::class, 'changePassword']);
        Route::post('/logout', [AuthController::class, 'logout']);
    });

    // ========== CUSTOMER ROUTES ==========
    Route::prefix('customer')->middleware('customer')->group(function () {
        // Menu & Promo
        Route::prefix('menu')->group(function () {
            Route::get('/categories', [CustomerMenuController::class, 'getCategories']);
            Route::get('/', [CustomerMenuController::class, 'index']);
            Route::get('/{id}', [CustomerMenuController::class, 'show']);
        });

        Route::get('/promos', [CustomerMenuController::class, 'getPromos']);

        // Cart
        Route::prefix('cart')->group(function () {
            Route::get('/', [CustomerCartController::class, 'index']);
            Route::post('/', [CustomerCartController::class, 'store']);
            Route::put('/{menu_id}', [CustomerCartController::class, 'update']);
            Route::delete('/{menu_id}', [CustomerCartController::class, 'destroy']);
            Route::delete('/', [CustomerCartController::class, 'clear']);
            Route::get('/check-checkout', [CustomerCartController::class, 'checkCheckout']);
        });

        // Orders
        Route::prefix('orders')->group(function () {
            Route::get('/checkout-summary', [CustomerOrderController::class, 'checkoutSummary']);
            Route::get('/', [CustomerOrderController::class, 'index']);
            Route::post('/', [CustomerOrderController::class, 'store']);
            Route::get('/{id}', [CustomerOrderController::class, 'show']);
            Route::post('/{id}/cancel', [CustomerOrderController::class, 'cancel']);
            Route::post('/{id}/reorder', [CustomerOrderController::class, 'reorder']);

            // Payments for order
            Route::post('/{order_id}/payments', [CustomerPaymentController::class, 'store']);

            // Reviews for order items
            Route::post('/{order_id}/items/{item_id}/review', [CustomerReviewController::class, 'store']);
        });

        // Payments
        Route::prefix('payments')->group(function () {
            Route::get('/', [CustomerPaymentController::class, 'index']);
            Route::get('/pending', [CustomerPaymentController::class, 'pendingPayments']);
            Route::get('/{id}', [CustomerPaymentController::class, 'show']);
            Route::get('/order/{order_id}', [CustomerPaymentController::class, 'getByOrder']);
            Route::post('/{id}/upload-proof', [CustomerPaymentController::class, 'uploadProof']);
            Route::post('/{id}/resend-notification', [CustomerPaymentController::class, 'resendNotification']);
            Route::get('/{id}/instructions', [CustomerPaymentController::class, 'getInstructions']);
        });

        Route::get('/payment-methods', [CustomerPaymentController::class, 'getPaymentMethods']);

        // Order payment creation
        Route::post('/orders/{order_id}/payments', [CustomerPaymentController::class, 'store']);

        // Payment methods
        Route::get('/payments/methods', [CustomerOrderController::class, 'getPaymentMethods']);


        // Reviews
        Route::prefix('reviews')->group(function () {
            Route::get('/', [CustomerReviewController::class, 'index']);
            Route::get('/pending', [CustomerReviewController::class, 'pendingReviews']);
            Route::get('/summary', [CustomerReviewController::class, 'summary']);
            Route::put('/{id}', [CustomerReviewController::class, 'update']);
            Route::delete('/{id}', [CustomerReviewController::class, 'destroy']);
        });

        Route::post('/orders/{order_id}/items/{item_id}/review', [CustomerReviewController::class, 'store']);
        Route::post('/orders/{order_id}/reviews/bulk', [CustomerReviewController::class, 'bulkStore']);

        // Public menu reviews
        Route::get('/menu/{menu_id}/reviews', [CustomerReviewController::class, 'getMenuReviews']);

        // Careers
        Route::prefix('careers')->group(function () {
            Route::get('/', [CustomerCareerController::class, 'index']);
            Route::get('/urgent', [CustomerCareerController::class, 'urgentJobs']);
            Route::get('/stats', [CustomerCareerController::class, 'careerStats']);
            Route::get('/tips', [CustomerCareerController::class, 'getApplicationTips']);
            Route::get('/{id}', [CustomerCareerController::class, 'show']);
            Route::post('/{id}/apply', [CustomerCareerController::class, 'apply']);

            // Application management
            Route::prefix('applications')->group(function () {
                Route::get('/my', [CustomerCareerController::class, 'myApplications']);
                Route::get('/{id}', [CustomerCareerController::class, 'getApplication']);
                Route::put('/{id}', [CustomerCareerController::class, 'updateApplication']);
                Route::delete('/{id}', [CustomerCareerController::class, 'withdrawApplication']);
                Route::get('/{id}/timeline', [CustomerCareerController::class, 'getApplicationTimeline']);
            });
        });
    });

    // ========== EMPLOYEE ROUTES ==========
    Route::prefix('employee')->middleware('auth:sanctum')->group(function () {
        // Auth routes (using existing AuthController with employee check)
        Route::prefix('auth')->group(function () {
            Route::get('/me', [EmployeeAuthController::class, 'me']);
            Route::post('/logout', [EmployeeAuthController::class, 'logout']);
            Route::get('/dashboard', [EmployeeAuthController::class, 'dashboard']);
        });

        // Order management
        Route::prefix('orders')->group(function () {
            Route::get('/', [EmployeeOrderController::class, 'index']);
            Route::get('/today', [EmployeeOrderController::class, 'todaysOrders']);
            Route::get('/status/{status}', [EmployeeOrderController::class, 'getOrdersByStatus']);
            Route::get('/statistics', [EmployeeOrderController::class, 'statistics']);
            Route::get('/{id}', [EmployeeOrderController::class, 'show']);
            Route::get('/{id}/receipt', [EmployeeOrderController::class, 'generateReceipt']);
            Route::put('/{id}/status', [EmployeeOrderController::class, 'updateStatus']);
            Route::post('/{id}/assign', [EmployeeOrderController::class, 'assignOrder']);
        });

        // Payment management
        Route::prefix('payments')->group(function () {
            Route::get('/', [EmployeePaymentController::class, 'index']);
            Route::get('/waiting-confirmation', [EmployeePaymentController::class, 'waitingConfirmation']);
            Route::get('/overdue', [EmployeePaymentController::class, 'overduePayments']);
            Route::get('/statistics', [EmployeePaymentController::class, 'statistics']);
            Route::get('/{id}', [EmployeePaymentController::class, 'show']);
            Route::post('/{id}/confirm', [EmployeePaymentController::class, 'confirm']);
            Route::post('/{id}/reject', [EmployeePaymentController::class, 'reject']);
        });
    });

    Route::post('/employee/auth/login', [\App\Http\Controllers\Api\Auth\AuthController::class, 'login']);

    // ========== ADMIN ROUTES ==========
    Route::prefix('admin')->middleware(['auth:sanctum', 'admin'])->group(function () {

        Route::get('/', [\App\Http\Controllers\Api\Admin\UserController::class, 'index']);
        Route::get('/statistics', [\App\Http\Controllers\Api\Admin\UserController::class, 'statistics']);
        Route::get('/{id}', [\App\Http\Controllers\Api\Admin\UserController::class, 'show']);
        Route::post('/', [\App\Http\Controllers\Api\Admin\UserController::class, 'store']);
        Route::put('/{id}', [\App\Http\Controllers\Api\Admin\UserController::class, 'update']);
        Route::delete('/{id}', [\App\Http\Controllers\Api\Admin\UserController::class, 'destroy']);
        Route::put('/{id}/toggle-status', [\App\Http\Controllers\Api\Admin\UserController::class, 'toggleStatus']);

        // Dashboard
        Route::get('/dashboard', [\App\Http\Controllers\Api\Admin\ReportController::class, 'dashboardReport']);

        // Menu Management
        Route::prefix('menus')->group(function () {
            Route::get('/', [\App\Http\Controllers\Api\Admin\MenuController::class, 'index']);
            Route::get('/statistics', [\App\Http\Controllers\Api\Admin\MenuController::class, 'statistics']);
            Route::get('/{id}', [\App\Http\Controllers\Api\Admin\MenuController::class, 'show']);
            Route::post('/', [\App\Http\Controllers\Api\Admin\MenuController::class, 'store']);
            Route::put('/{id}', [\App\Http\Controllers\Api\Admin\MenuController::class, 'update']);
            Route::delete('/{id}', [\App\Http\Controllers\Api\Admin\MenuController::class, 'destroy']);
            Route::post('/bulk-update-status', [\App\Http\Controllers\Api\Admin\MenuController::class, 'bulkUpdateStatus']);
        });

        // Category Management
        Route::prefix('categories')->group(function () {
            Route::get('/', [\App\Http\Controllers\Api\Admin\CategoryController::class, 'index']);
            Route::get('/{id}', [\App\Http\Controllers\Api\Admin\CategoryController::class, 'show']);
            Route::post('/', [\App\Http\Controllers\Api\Admin\CategoryController::class, 'store']);
            Route::put('/{id}', [\App\Http\Controllers\Api\Admin\CategoryController::class, 'update']);
            Route::delete('/{id}', [\App\Http\Controllers\Api\Admin\CategoryController::class, 'destroy']);
            Route::put('/{id}/toggle-status', [\App\Http\Controllers\Api\Admin\CategoryController::class, 'toggleStatus']);
        });

        // Promo Management
        Route::prefix('promos')->group(function () {
            Route::get('/', [\App\Http\Controllers\Api\Admin\PromoController::class, 'index']);
            Route::get('/statistics', [\App\Http\Controllers\Api\Admin\PromoController::class, 'statistics']);
            Route::get('/{id}', [\App\Http\Controllers\Api\Admin\PromoController::class, 'show']);
            Route::post('/', [\App\Http\Controllers\Api\Admin\PromoController::class, 'store']);
            Route::put('/{id}', [\App\Http\Controllers\Api\Admin\PromoController::class, 'update']);
            Route::delete('/{id}', [\App\Http\Controllers\Api\Admin\PromoController::class, 'destroy']);
            Route::put('/{id}/toggle-status', [\App\Http\Controllers\Api\Admin\PromoController::class, 'toggleStatus']);
        });

        // Payment Method Management
        Route::prefix('payment-methods')->group(function () {
            Route::get('/', [\App\Http\Controllers\Api\Admin\PaymentMethodController::class, 'index']);
            Route::get('/statistics', [\App\Http\Controllers\Api\Admin\PaymentMethodController::class, 'statistics']);
            Route::get('/{id}', [\App\Http\Controllers\Api\Admin\PaymentMethodController::class, 'show']);
            Route::post('/', [\App\Http\Controllers\Api\Admin\PaymentMethodController::class, 'store']);
            Route::put('/{id}', [\App\Http\Controllers\Api\Admin\PaymentMethodController::class, 'update']);
            Route::delete('/{id}', [\App\Http\Controllers\Api\Admin\PaymentMethodController::class, 'destroy']);
            Route::put('/{id}/toggle-status', [\App\Http\Controllers\Api\Admin\PaymentMethodController::class, 'toggleStatus']);
        });

        // Career Management
        Route::prefix('careers')->group(function () {
            // Job Postings
            Route::get('/', [\App\Http\Controllers\Api\Admin\CareerController::class, 'index']);
            Route::get('/statistics', [\App\Http\Controllers\Api\Admin\CareerController::class, 'statistics']);
            Route::get('/{id}', [\App\Http\Controllers\Api\Admin\CareerController::class, 'show']);
            Route::post('/', [\App\Http\Controllers\Api\Admin\CareerController::class, 'store']);
            Route::put('/{id}', [\App\Http\Controllers\Api\Admin\CareerController::class, 'update']);
            Route::delete('/{id}', [\App\Http\Controllers\Api\Admin\CareerController::class, 'destroy']);
            Route::put('/{id}/toggle-status', [\App\Http\Controllers\Api\Admin\CareerController::class, 'toggleStatus']);

            // Job Applications
            Route::get('/applications', [\App\Http\Controllers\Api\Admin\CareerController::class, 'applications']);
            Route::get('/applications/{id}', [\App\Http\Controllers\Api\Admin\CareerController::class, 'showApplication']);
            Route::put('/applications/{id}/status', [\App\Http\Controllers\Api\Admin\CareerController::class, 'updateApplicationStatus']);
        });

        // System Settings
        Route::prefix('system-settings')->group(function () {
            Route::get('/', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'index']);
            Route::get('/statistics', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'dashboardStatistics']);
            Route::get('/{key}', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'showByKey']);
            Route::post('/', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'store']);
            Route::put('/{id}', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'update']);
            Route::delete('/{id}', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'destroy']);
            Route::post('/bulk-update', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'bulkUpdate']);

            // Operational Hours
            Route::get('/operational-hours', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'getOperationalHours']);
            Route::put('/operational-hours', [\App\Http\Controllers\Api\Admin\SystemSettingsController::class, 'updateOperationalHours']);
        });

        // Reports
        Route::prefix('reports')->group(function () {
            Route::get('/sales', [\App\Http\Controllers\Api\Admin\ReportController::class, 'salesReport']);
            Route::get('/customers', [\App\Http\Controllers\Api\Admin\ReportController::class, 'customerReport']);
            Route::get('/inventory', [\App\Http\Controllers\Api\Admin\ReportController::class, 'inventoryReport']);
        });

        // Order Management
        Route::prefix('orders')->group(function () {
            Route::get('/', [\App\Http\Controllers\Api\Admin\OrderController::class, 'index']);
            Route::get('/{id}', [\App\Http\Controllers\Api\Admin\OrderController::class, 'show']);
            Route::put('/{id}/status', [\App\Http\Controllers\Api\Admin\OrderController::class, 'updateStatus']);
            Route::get('/{id}/receipt', [\App\Http\Controllers\Api\Admin\OrderController::class, 'generateReceipt']);
        });

        // User Management
        Route::prefix('users')->group(function () {
            Route::get('/', [\App\Http\Controllers\Api\Admin\UserController::class, 'index']);
            Route::get('/statistics', [\App\Http\Controllers\Api\Admin\UserController::class, 'statistics']);
            Route::get('/{id}', [\App\Http\Controllers\Api\Admin\UserController::class, 'show']);
            Route::post('/', [\App\Http\Controllers\Api\Admin\UserController::class, 'store']);
            Route::put('/{id}', [\App\Http\Controllers\Api\Admin\UserController::class, 'update']);
            Route::delete('/{id}', [\App\Http\Controllers\Api\Admin\UserController::class, 'destroy']);
            Route::put('/{id}/toggle-status', [\App\Http\Controllers\Api\Admin\UserController::class, 'toggleStatus']);
        });
    });
});

// Fallback for API 404
Route::fallback(function () {
    return response()->json([
        'success' => false,
        'message' => 'API endpoint not found',
        'errors' => ['Endpoint tidak ditemukan']
    ], 404);
});
