{{-- resources/views/employee/receipts/index.blade.php --}}
@extends('layouts.employee')

@section('title', 'Print Receipts')
@section('page_title', 'Print Receipts')
@section('icon', 'fa-receipt')

@section('breadcrumb')
    <li class="breadcrumb-item active">Print Receipts</li>
@endsection

@section('content')
<div class="container-fluid">
    <!-- Receipt Generator -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="card employee-card">
                <div class="employee-card-header">
                    <h6 class="mb-0">
                        <i class="fas fa-print me-2"></i>Receipt Generator
                    </h6>
                </div>
                <div class="card-body">
                    <form method="POST" action="{{ route('employee.receipts.generate') }}" target="_blank">
                        @csrf
                        
                        <div class="row g-3">
                            <div class="col-md-6">
                                <label for="order_number" class="form-label">Find Order</label>
                                <div class="input-group">
                                    <input type="text" class="form-control" 
                                           id="order_number" name="order_number" 
                                           placeholder="Enter order number (e.g., ORD-2024-001)">
                                    <button class="btn btn-outline-secondary" type="button" id="find-order">
                                        <i class="fas fa-search"></i>
                                    </button>
                                </div>
                                <small class="text-muted">Enter order number to load details</small>
                            </div>
                            
                            <div class="col-md-6">
                                <label for="receipt_type" class="form-label">Receipt Type</label>
                                <select class="form-control" id="receipt_type" name="receipt_type">
                                    <option value="customer">Customer Receipt</option>
                                    <option value="kitchen">Kitchen Ticket</option>
                                    <option value="payment">Payment Receipt</option>
                                    <option value="refund">Refund Receipt</option>
                                </select>
                            </div>
                        </div>
                        
                        <!-- Order Details (will be populated by JavaScript) -->
                        <div id="order-details" class="mt-4" style="display: none;">
                            <div class="alert alert-info">
                                <div class="row">
                                    <div class="col-md-4">
                                        <strong>Order #:</strong> <span id="order-id"></span><br>
                                        <strong>Customer:</strong> <span id="order-customer"></span>
                                    </div>
                                    <div class="col-md-4">
                                        <strong>Total Amount:</strong> <span id="order-total"></span><br>
                                        <strong>Status:</strong> <span id="order-status"></span>
                                    </div>
                                    <div class="col-md-4">
                                        <strong>Date:</strong> <span id="order-date"></span><br>
                                        <strong>Items:</strong> <span id="order-items"></span>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Receipt Options -->
                        <div class="row g-3 mt-3">
                            <div class="col-md-4">
                                <label class="form-label">Print Options</label>
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" id="print_header" name="print_header" checked>
                                    <label class="form-check-label" for="print_header">
                                        Print Restaurant Header
                                    </label>
                                </div>
                                <div class="form-check mb-2">
                                    <input class="form-check-input" type="checkbox" id="print_footer" name="print_footer" checked>
                                    <label class="form-check-label" for="print_footer">
                                        Print Footer & Thank You Message
                                    </label>
                                </div>
                                <div class="form-check">
                                    <input class="form-check-input" type="checkbox" id="print_qr" name="print_qr">
                                    <label class="form-check-label" for="print_qr">
                                        Include QR Code for Feedback
                                    </label>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <label for="copies" class="form-label">Number of Copies</label>
                                <select class="form-control" id="copies" name="copies">
                                    @for($i = 1; $i <= 5; $i++)
                                        <option value="{{ $i }}">{{ $i }} copy{{ $i > 1 ? 'ies' : '' }}</option>
                                    @endfor
                                </select>
                            </div>
                            
                            <div class="col-md-4 d-flex align-items-end">
                                <div class="d-flex gap-2 w-100">
                                    <button type="submit" class="btn btn-primary flex-grow-1">
                                        <i class="fas fa-print me-2"></i> Print Receipt
                                    </button>
                                    <button type="button" class="btn btn-secondary" id="preview-receipt">
                                        <i class="fas fa-eye me-2"></i> Preview
                                    </button>
                                </div>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    </div>
    
    <!-- Recent Receipts -->
    <div class="row">
        <div class="col-12">
            <div class="card employee-card">
                <div class="employee-card-header d-flex justify-content-between align-items-center">
                    <h6 class="mb-0">
                        <i class="fas fa-history me-2"></i>Recently Printed Receipts
                    </h6>
                    <button class="btn btn-sm btn-outline-primary" onclick="printRecentReceipts()">
                        <i class="fas fa-print me-1"></i> Print All Today
                    </button>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table table-hover employee-table">
                            <thead>
                                <tr>
                                    <th>Receipt #</th>
                                    <th>Order #</th>
                                    <th>Type</th>
                                    <th>Amount</th>
                                    <th>Printed By</th>
                                    <th>Printed At</th>
                                    <th>Copies</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                @forelse($receipts as $receipt)
                                <tr>
                                    <td><strong>{{ $receipt->receipt_number }}</strong></td>
                                    <td>
                                        <a href="{{ route('employee.orders.show', $receipt->order_id) }}" 
                                           class="text-decoration-none">
                                            #{{ $receipt->order->order_number ?? 'N/A' }}
                                        </a>
                                    </td>
                                    <td>
                                        <span class="badge {{ $receipt->type == 'customer' ? 'bg-primary' : ($receipt->type == 'kitchen' ? 'bg-warning' : 'bg-info') }}">
                                            {{ ucfirst($receipt->type) }}
                                        </span>
                                    </td>
                                    <td>Rp {{ number_format($receipt->amount, 0, ',', '.') }}</td>
                                    <td>{{ $receipt->printedBy->name ?? auth()->user()->name }}</td>
                                    <td>
                                        <small class="text-muted">
                                            {{ $receipt->created_at->format('d M Y, H:i') }}<br>
                                            {{ $receipt->created_at->diffForHumans() }}
                                        </small>
                                    </td>
                                    <td>{{ $receipt->copies }}</td>
                                    <td>
                                        <div class="btn-group btn-group-sm" role="group">
                                            <a href="{{ route('employee.receipts.reprint', $receipt->id) }}" 
                                               class="btn btn-primary" target="_blank"
                                               data-bs-toggle="tooltip" title="Reprint">
                                                <i class="fas fa-print"></i>
                                            </a>
                                            <a href="{{ route('employee.receipts.download', $receipt->id) }}" 
                                               class="btn btn-info" 
                                               data-bs-toggle="tooltip" title="Download PDF">
                                                <i class="fas fa-download"></i>
                                            </a>
                                            <button type="button" class="btn btn-warning" 
                                                    onclick="showReceiptDetails({{ $receipt->id }})"
                                                    data-bs-toggle="tooltip" title="View Details">
                                                <i class="fas fa-eye"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                @empty
                                <tr>
                                    <td colspan="8" class="text-center py-4">
                                        <i class="fas fa-receipt fa-3x text-muted mb-3"></i>
                                        <h5>No receipts printed today</h5>
                                        <p class="text-muted">Start by printing a receipt above</p>
                                    </td>
                                </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Receipt Details Modal -->
<div class="modal fade" id="receiptDetailsModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Receipt Details</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body" id="receipt-details-content">
                <!-- Content will be loaded here -->
            </div>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Find order by number
        document.getElementById('find-order').addEventListener('click', findOrder);
        document.getElementById('order_number').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                e.preventDefault();
                findOrder();
            }
        });
        
        // Preview receipt
        document.getElementById('preview-receipt').addEventListener('click', function() {
            const orderNumber = document.getElementById('order_number').value;
            if (orderNumber) {
                window.open(`/employee/receipts/preview?order_number=${orderNumber}`, '_blank');
            } else {
                alert('Please enter an order number first');
            }
        });
    });
    
    function findOrder() {
        const orderNumber = document.getElementById('order_number').value;
        if (!orderNumber) {
            alert('Please enter an order number');
            return;
        }
        
        // In a real app, this would be an API call
        // For demo, using sample data
        const sampleOrder = {
            id: 'ORD-2024-001',
            customer: 'John Doe',
            total: 'Rp 185,000',
            status: 'completed',
            date: '15 Jan 2024, 14:30',
            items: 3
        };
        
        document.getElementById('order-id').textContent = sampleOrder.id;
        document.getElementById('order-customer').textContent = sampleOrder.customer;
        document.getElementById('order-total').textContent = sampleOrder.total;
        document.getElementById('order-status').textContent = sampleOrder.status;
        document.getElementById('order-date').textContent = sampleOrder.date;
        document.getElementById('order-items').textContent = sampleOrder.items;
        
        document.getElementById('order-details').style.display = 'block';
    }
    
    function showReceiptDetails(receiptId) {
        // In a real app, this would be an API call
        fetch(`/employee/receipts/${receiptId}/details`)
            .then(response => response.json())
            .then(data => {
                const modal = new bootstrap.Modal(document.getElementById('receiptDetailsModal'));
                document.getElementById('receipt-details-content').innerHTML = data.html;
                modal.show();
            })
            .catch(error => {
                console.error('Error:', error);
                alert('Failed to load receipt details');
            });
    }
    
    function printRecentReceipts() {
        if (confirm('Print all receipts from today?')) {
            window.open('/employee/receipts/print-today', '_blank');
        }
    }
</script>
@endsection