@extends('admin.layouts.main')

@section('title', 'Orders by Date Range')

@section('content')
<div class="container-fluid">
    <div class="d-sm-flex align-items-center justify-content-between mb-4">
        <h1 class="h3 mb-0 text-gray-800">Orders by Date Range</h1>
        <div class="d-flex">
            <button class="btn btn-success mr-2" onclick="exportToExcel()">
                <i class="fas fa-file-excel"></i> Export
            </button>
        </div>
    </div>

    <!-- Date Range Filter -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Select Date Range</h6>
        </div>
        <div class="card-body">
            <form id="dateRangeForm" method="GET" action="{{ route('admin.reports.orders.date-range') }}">
                <div class="row">
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="start_date">Start Date</label>
                            <input type="date" class="form-control" id="start_date" name="start_date" 
                                   value="{{ request('start_date', date('Y-m-d', strtotime('-7 days'))) }}">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="end_date">End Date</label>
                            <input type="date" class="form-control" id="end_date" name="end_date" 
                                   value="{{ request('end_date', date('Y-m-d')) }}">
                        </div>
                    </div>
                    <div class="col-md-3">
                        <div class="form-group">
                            <label for="status">Order Status</label>
                            <select class="form-control" id="status" name="status">
                                <option value="">All Status</option>
                                <option value="pending" {{ request('status') == 'pending' ? 'selected' : '' }}>Pending</option>
                                <option value="processing" {{ request('status') == 'processing' ? 'selected' : '' }}>Processing</option>
                                <option value="completed" {{ request('status') == 'completed' ? 'selected' : '' }}>Completed</option>
                                <option value="cancelled" {{ request('status') == 'cancelled' ? 'selected' : '' }}>Cancelled</option>
                            </select>
                        </div>
                    </div>
                    <div class="col-md-3 d-flex align-items-end">
                        <button type="submit" class="btn btn-primary mr-2">
                            <i class="fas fa-filter"></i> Filter
                        </button>
                        <button type="button" class="btn btn-secondary" onclick="resetFilters()">
                            <i class="fas fa-redo"></i> Reset
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Summary Statistics -->
    <div class="row">
        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-primary shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">
                                Total Orders
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                {{ isset($apiData['total_orders']) ? $apiData['total_orders'] : 0 }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-shopping-cart fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-success shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-success text-uppercase mb-1">
                                Total Revenue
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                Rp {{ number_format(isset($apiData['total_revenue']) ? $apiData['total_revenue'] : 0, 0, ',', '.') }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-dollar-sign fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-info shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-info text-uppercase mb-1">
                                Average Order Value
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                Rp {{ number_format(isset($apiData['average_order_value']) ? $apiData['average_order_value'] : 0, 0, ',', '.') }}
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-chart-bar fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="col-xl-3 col-md-6 mb-4">
            <div class="card border-left-warning shadow h-100 py-2">
                <div class="card-body">
                    <div class="row no-gutters align-items-center">
                        <div class="col mr-2">
                            <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">
                                Peak Day
                            </div>
                            <div class="h5 mb-0 font-weight-bold text-gray-800">
                                {{ isset($apiData['peak_day']['date']) ? date('d M', strtotime($apiData['peak_day']['date'])) : 'N/A' }}
                            </div>
                            <div class="mt-1 text-xs">
                                {{ isset($apiData['peak_day']['orders']) ? $apiData['peak_day']['orders'] : 0 }} orders
                            </div>
                        </div>
                        <div class="col-auto">
                            <i class="fas fa-calendar-day fa-2x text-gray-300"></i>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Daily Orders Chart -->
    <div class="card shadow mb-4">
        <div class="card-header py-3">
            <h6 class="m-0 font-weight-bold text-primary">Daily Orders Breakdown</h6>
        </div>
        <div class="card-body">
            <canvas id="dailyOrdersChart"></canvas>
        </div>
    </div>

    <!-- Orders Table -->
    <div class="card shadow mb-4">
        <div class="card-header py-3 d-flex justify-content-between align-items-center">
            <h6 class="m-0 font-weight-bold text-primary">Order Details</h6>
            <div class="text-muted">
                Showing {{ isset($apiData['orders']) ? count($apiData['orders']) : 0 }} orders
            </div>
        </div>
        <div class="card-body">
            <div class="table-responsive">
                <table class="table table-bordered" id="ordersTable" width="100%" cellspacing="0">
                    <thead>
                        <tr>
                            <th>Order ID</th>
                            <th>Date</th>
                            <th>Customer</th>
                            <th>Items</th>
                            <th>Total (Rp)</th>
                            <th>Payment Method</th>
                            <th>Status</th>
                            <th>Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                        @if(isset($apiData['orders']) && count($apiData['orders']) > 0)
                            @foreach($apiData['orders'] as $order)
                            <tr>
                                <td>#{{ $order['order_id'] }}</td>
                                <td>{{ date('d/m/Y H:i', strtotime($order['created_at'])) }}</td>
                                <td>{{ $order['customer_name'] }}</td>
                                <td>{{ $order['item_count'] }}</td>
                                <td>Rp {{ number_format($order['total_amount'], 0, ',', '.') }}</td>
                                <td>
                                    <span class="badge bg-info">
                                        {{ ucfirst(str_replace('_', ' ', $order['payment_method'])) }}
                                    </span>
                                </td>
                                <td>
                                    <span class="badge bg-{{ $order['status'] == 'completed' ? 'success' : ($order['status'] == 'pending' ? 'warning' : ($order['status'] == 'processing' ? 'info' : 'secondary')) }}">
                                        {{ ucfirst($order['status']) }}
                                    </span>
                                </td>
                                <td>
                                    <a href="{{ route('employee.orders.show', $order['order_id']) }}" 
                                       class="btn btn-sm btn-primary" target="_blank">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                </td>
                            </tr>
                            @endforeach
                        @else
                            <tr>
                                <td colspan="8" class="text-center">No orders found for the selected date range</td>
                            </tr>
                        @endif
                    </tbody>
                </table>
            </div>
        </div>
    </div>

    <!-- Additional Statistics -->
    <div class="row">
        <div class="col-lg-6">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Order Status Distribution</h6>
                </div>
                <div class="card-body">
                    <canvas id="statusChart"></canvas>
                </div>
            </div>
        </div>

        <div class="col-lg-6">
            <div class="card shadow mb-4">
                <div class="card-header py-3">
                    <h6 class="m-0 font-weight-bold text-primary">Top Customers</h6>
                </div>
                <div class="card-body">
                    <div class="table-responsive">
                        <table class="table">
                            <thead>
                                <tr>
                                    <th>Customer</th>
                                    <th>Orders</th>
                                    <th>Total Spent</th>
                                    <th>Avg. Order</th>
                                </tr>
                            </thead>
                            <tbody>
                                @if(isset($apiData['top_customers']) && count($apiData['top_customers']) > 0)
                                    @foreach($apiData['top_customers'] as $customer)
                                    <tr>
                                        <td>
                                            <div class="d-flex align-items-center">
                                                <div class="mr-3">
                                                    <img src="https://ui-avatars.com/api/?name={{ urlencode($customer['name']) }}&background=4e73df&color=fff" 
                                                         alt="{{ $customer['name'] }}" 
                                                         style="width: 30px; height: 30px; border-radius: 50%;">
                                                </div>
                                                <div>
                                                    <div class="font-weight-bold">{{ $customer['name'] }}</div>
                                                    <small class="text-muted">{{ $customer['email'] }}</small>
                                                </div>
                                            </div>
                                        </td>
                                        <td>{{ $customer['order_count'] }}</td>
                                        <td>Rp {{ number_format($customer['total_spent'], 0, ',', '.') }}</td>
                                        <td>Rp {{ number_format($customer['average_order'], 0, ',', '.') }}</td>
                                    </tr>
                                    @endforeach
                                @else
                                    <tr>
                                        <td colspan="4" class="text-center">No customer data available</td>
                                    </tr>
                                @endif
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Initialize DataTable
        $('#ordersTable').DataTable({
            pageLength: 25,
            order: [[1, 'desc']],
            responsive: true
        });

        // Daily Orders Chart
        const dailyOrdersCtx = document.getElementById('dailyOrdersChart').getContext('2d');
        const dailyData = @json(isset($apiData['daily_orders']) ? $apiData['daily_orders'] : []);
        
        new Chart(dailyOrdersCtx, {
            type: 'bar',
            data: {
                labels: dailyData.map(item => item.date),
                datasets: [{
                    label: 'Number of Orders',
                    data: dailyData.map(item => item.count),
                    backgroundColor: 'rgba(78, 115, 223, 0.8)',
                    borderColor: 'rgba(78, 115, 223, 1)',
                    borderWidth: 1
                }, {
                    label: 'Revenue (Rp)',
                    data: dailyData.map(item => item.revenue / 1000000), // Convert to millions for scale
                    backgroundColor: 'rgba(28, 200, 138, 0.8)',
                    borderColor: 'rgba(28, 200, 138, 1)',
                    borderWidth: 1,
                    yAxisID: 'y1',
                    type: 'line'
                }]
            },
            options: {
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: 'Number of Orders'
                        }
                    },
                    y1: {
                        beginAtZero: true,
                        position: 'right',
                        title: {
                            display: true,
                            text: 'Revenue (Millions Rp)'
                        },
                        grid: {
                            drawOnChartArea: false
                        },
                        ticks: {
                            callback: function(value) {
                                return 'Rp ' + value + 'M';
                            }
                        }
                    }
                }
            }
        });

        // Status Distribution Chart
        const statusCtx = document.getElementById('statusChart').getContext('2d');
        const statusData = @json(isset($apiData['status_distribution']) ? $apiData['status_distribution'] : []);
        
        new Chart(statusCtx, {
            type: 'doughnut',
            data: {
                labels: statusData.map(item => item.status),
                datasets: [{
                    data: statusData.map(item => item.count),
                    backgroundColor: [
                        '#4e73df', // Pending
                        '#36b9cc', // Processing
                        '#1cc88a', // Completed
                        '#f6c23e', // Cancelled
                        '#e74a3b'  // Failed
                    ]
                }]
            },
            options: {
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    });

    function exportToExcel() {
        // Get current filter values
        const startDate = document.getElementById('start_date').value;
        const endDate = document.getElementById('end_date').value;
        const status = document.getElementById('status').value;
        
        // Create export URL
        const exportUrl = `{{ route('admin.reports.orders.date-range') }}?export=excel&start_date=${startDate}&end_date=${endDate}&status=${status}`;
        
        // Redirect to export URL
        window.location.href = exportUrl;
    }

    function resetFilters() {
        document.getElementById('start_date').value = '{{ date('Y-m-d', strtotime('-7 days')) }}';
        document.getElementById('end_date').value = '{{ date('Y-m-d') }}';
        document.getElementById('status').value = '';
        document.getElementById('dateRangeForm').submit();
    }
</script>
@endsection