{{-- resources/views/admin/promos/create.blade.php --}}
@extends('layouts.admin')

@section('title', 'Add Promotion')
@section('page_title', 'Add Promotion')
@section('icon', 'fa-tag')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.promos.index') }}">Promo Management</a></li>
    <li class="breadcrumb-item active">Add Promotion</li>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row justify-content-center">
        <div class="col-lg-10">
            <div class="card">
                <div class="card-header">
                    <div class="d-flex justify-content-between align-items-center">
                        <h5 class="card-title mb-0">
                            <i class="fas fa-tag me-2"></i>Add New Promotion
                        </h5>
                        <a href="{{ route('admin.promos.index') }}" class="btn btn-secondary btn-sm">
                            <i class="fas fa-arrow-left me-1"></i> Back to List
                        </a>
                    </div>
                </div>
                
                <form action="{{ route('admin.promos.store') }}" method="POST">
                    @csrf
                    
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-8">
                                <!-- Basic Information -->
                                <div class="card mb-4">
                                    <div class="card-header">
                                        <h6 class="card-title mb-0">
                                            <i class="fas fa-info-circle me-2"></i>Basic Information
                                        </h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-md-12 mb-3">
                                                <label for="name" class="form-label">Promotion Name *</label>
                                                <input type="text" class="form-control @error('name') is-invalid @enderror" 
                                                       id="name" name="name" value="{{ old('name') }}" required>
                                                @error('name')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @enderror
                                            </div>
                                        </div>
                                        
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label for="type" class="form-label">Discount Type *</label>
                                                <select class="form-control @error('type') is-invalid @enderror" 
                                                        id="type" name="type" required>
                                                    <option value="">Select Type</option>
                                                    <option value="percentage" {{ old('type') == 'percentage' ? 'selected' : '' }}>
                                                        Percentage Discount
                                                    </option>
                                                    <option value="fixed" {{ old('type') == 'fixed' ? 'selected' : '' }}>
                                                        Fixed Amount Discount
                                                    </option>
                                                    <option value="bogo" {{ old('type') == 'bogo' ? 'selected' : '' }}>
                                                        Buy One Get One (BOGO)
                                                    </option>
                                                </select>
                                                @error('type')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @enderror
                                            </div>
                                            
                                            <div class="col-md-6 mb-3">
                                                <label for="discount_value" class="form-label">Discount Value *</label>
                                                <div class="input-group">
                                                    <input type="number" class="form-control @error('discount_value') is-invalid @enderror" 
                                                           id="discount_value" name="discount_value" 
                                                           value="{{ old('discount_value') }}" 
                                                           min="0" step="0.01" required>
                                                    <span class="input-group-text" id="discount_suffix">%</span>
                                                    @error('discount_value')
                                                        <div class="invalid-feedback">{{ $message }}</div>
                                                    @enderror
                                                </div>
                                                <small class="form-text text-muted" id="discount_hint">
                                                    Enter percentage (e.g., 10 for 10%)
                                                </small>
                                            </div>
                                        </div>
                                        
                                        <div class="row">
                                            <div class="col-md-6 mb-3">
                                                <label for="code" class="form-label">Promo Code</label>
                                                <input type="text" class="form-control @error('code') is-invalid @enderror" 
                                                       id="code" name="code" value="{{ old('code') }}" 
                                                       placeholder="e.g., SUMMER2024">
                                                @error('code')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @enderror
                                                <small class="form-text text-muted">
                                                    Leave blank for auto-apply promotions
                                                </small>
                                            </div>
                                            
                                            <div class="col-md-6 mb-3">
                                                <label for="max_discount" class="form-label">Maximum Discount (Rp)</label>
                                                <input type="number" class="form-control @error('max_discount') is-invalid @enderror" 
                                                       id="max_discount" name="max_discount" 
                                                       value="{{ old('max_discount') }}" 
                                                       min="0" step="1000" placeholder="Optional">
                                                @error('max_discount')
                                                    <div class="invalid-feedback">{{ $message }}</div>
                                                @enderror
                                                <small class="form-text text-muted">
                                                    Maximum discount amount (for percentage discounts)
                                                </small>
                                            </div>
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="description" class="form-label">Description</label>
                                            <textarea class="form-control @error('description') is-invalid @enderror" 
                                                      id="description" name="description" rows="3">{{ old('description') }}</textarea>
                                            @error('description')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-4">
                                <!-- Validity Period -->
                                <div class="card mb-4">
                                    <div class="card-header">
                                        <h6 class="card-title mb-0">
                                            <i class="fas fa-calendar me-2"></i>Validity Period
                                        </h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="mb-3">
                                            <label for="start_date" class="form-label">Start Date *</label>
                                            <input type="date" class="form-control @error('start_date') is-invalid @enderror" 
                                                   id="start_date" name="start_date" 
                                                   value="{{ old('start_date', date('Y-m-d')) }}" required>
                                            @error('start_date')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="end_date" class="form-label">End Date *</label>
                                            <input type="date" class="form-control @error('end_date') is-invalid @enderror" 
                                                   id="end_date" name="end_date" 
                                                   value="{{ old('end_date', date('Y-m-d', strtotime('+1 month'))) }}" required>
                                            @error('end_date')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                        </div>
                                        
                                        <div class="mb-3">
                                            <label for="min_purchase" class="form-label">Minimum Purchase (Rp)</label>
                                            <input type="number" class="form-control @error('min_purchase') is-invalid @enderror" 
                                                   id="min_purchase" name="min_purchase" 
                                                   value="{{ old('min_purchase') }}" 
                                                   min="0" step="1000" placeholder="Optional">
                                            @error('min_purchase')
                                                <div class="invalid-feedback">{{ $message }}</div>
                                            @enderror
                                            <small class="form-text text-muted">
                                                Minimum order amount to apply this promo
                                            </small>
                                        </div>
                                        
                                        <div class="form-check form-switch mb-3">
                                            <input class="form-check-input" type="checkbox" 
                                                   id="is_active" name="is_active" 
                                                   value="1" {{ old('is_active', true) ? 'checked' : '' }}>
                                            <label class="form-check-label" for="is_active">
                                                Active Promotion
                                            </label>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Preview -->
                                <div class="card">
                                    <div class="card-header">
                                        <h6 class="card-title mb-0">
                                            <i class="fas fa-eye me-2"></i>Preview
                                        </h6>
                                    </div>
                                    <div class="card-body">
                                        <div class="alert alert-info mb-0">
                                            <h6 class="alert-heading">Promotion Preview</h6>
                                            <hr class="my-2">
                                            <p class="mb-1">
                                                <strong>Name:</strong> 
                                                <span id="preview_name">{{ old('name', 'New Promotion') }}</span>
                                            </p>
                                            <p class="mb-1">
                                                <strong>Type:</strong> 
                                                <span id="preview_type">{{ old('type', 'Percentage Discount') }}</span>
                                            </p>
                                            <p class="mb-1">
                                                <strong>Value:</strong> 
                                                <span id="preview_value">{{ old('discount_value', 0) }}</span>
                                            </p>
                                            <p class="mb-0">
                                                <strong>Valid:</strong> 
                                                <span id="preview_dates">
                                                    {{ old('start_date', date('d M Y')) }} - {{ old('end_date', date('d M Y', strtotime('+1 month'))) }}
                                                </span>
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="card-footer">
                        <div class="d-flex justify-content-between">
                            <button type="reset" class="btn btn-secondary">
                                <i class="fas fa-redo me-1"></i> Reset
                            </button>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save me-1"></i> Create Promotion
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Elements
        const typeSelect = document.getElementById('type');
        const discountValueInput = document.getElementById('discount_value');
        const discountSuffix = document.getElementById('discount_suffix');
        const discountHint = document.getElementById('discount_hint');
        const nameInput = document.getElementById('name');
        const startDateInput = document.getElementById('start_date');
        const endDateInput = document.getElementById('end_date');
        
        // Preview elements
        const previewName = document.getElementById('preview_name');
        const previewType = document.getElementById('preview_type');
        const previewValue = document.getElementById('preview_value');
        const previewDates = document.getElementById('preview_dates');
        
        // Update preview when name changes
        nameInput.addEventListener('input', function() {
            previewName.textContent = this.value || 'New Promotion';
        });
        
        // Update when discount type changes
        typeSelect.addEventListener('change', function() {
            const type = this.value;
            let suffix = '%';
            let hint = 'Enter percentage (e.g., 10 for 10%)';
            let typeText = 'Percentage Discount';
            
            if (type === 'fixed') {
                suffix = 'Rp';
                hint = 'Enter fixed amount (e.g., 5000 for Rp 5,000)';
                typeText = 'Fixed Amount';
            } else if (type === 'bogo') {
                suffix = '';
                discountValueInput.value = '';
                discountValueInput.disabled = true;
                hint = 'Buy One Get One - No value needed';
                typeText = 'Buy One Get One';
            } else {
                discountValueInput.disabled = false;
            }
            
            discountSuffix.textContent = suffix;
            discountHint.textContent = hint;
            previewType.textContent = typeText;
            
            updateValuePreview();
        });
        
        // Update preview when discount value changes
        discountValueInput.addEventListener('input', updateValuePreview);
        
        function updateValuePreview() {
            const type = typeSelect.value;
            const value = discountValueInput.value;
            
            if (type === 'percentage') {
                previewValue.textContent = value + '% discount';
            } else if (type === 'fixed') {
                previewValue.textContent = 'Rp ' + Number(value).toLocaleString('id-ID') + ' discount';
            } else if (type === 'bogo') {
                previewValue.textContent = 'Buy One Get One';
            }
        }
        
        // Update date preview
        function updateDatePreview() {
            const startDate = startDateInput.value ? new Date(startDateInput.value) : new Date();
            const endDate = endDateInput.value ? new Date(endDateInput.value) : new Date();
            
            const formatDate = (date) => {
                return date.toLocaleDateString('id-ID', {
                    day: '2-digit',
                    month: 'short',
                    year: 'numeric'
                });
            };
            
            previewDates.textContent = formatDate(startDate) + ' - ' + formatDate(endDate);
        }
        
        startDateInput.addEventListener('change', updateDatePreview);
        endDateInput.addEventListener('change', updateDatePreview);
        
        // Set min date for end date
        startDateInput.addEventListener('change', function() {
            endDateInput.min = this.value;
        });
        
        // Set max date for start date
        endDateInput.addEventListener('change', function() {
            startDateInput.max = this.value;
        });
        
        // Initialize previews
        updateDatePreview();
        updateValuePreview();
    });
</script>
@endsection