@extends('admin.layouts.main')

@section('title', 'Payment Method Details')
@section('page-title', 'Payment Method Details')

@section('breadcrumbs')
    <li class="breadcrumb-item"><a href="{{ route('admin.payment-methods.index') }}">Payment Methods</a></li>
    <li class="breadcrumb-item active">{{ $paymentMethod['name'] }}</li>
@endsection

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-lg-8">
            <div class="card">
                <div class="card-header d-flex justify-content-between align-items-center">
                    <h6 class="card-title mb-0">Payment Method Information</h6>
                    <div>
                        <span class="badge badge-{{ $paymentMethod['is_active'] ? 'success' : 'danger' }}">
                            {{ $paymentMethod['is_active'] ? 'Active' : 'Inactive' }}
                        </span>
                        <span class="badge bg-{{ getPaymentMethodColor($paymentMethod['code']) }} ml-2">
                            {{ ucfirst(str_replace('_', ' ', $paymentMethod['code'])) }}
                        </span>
                    </div>
                </div>
                <div class="card-body">
                    <div class="row">
                        <div class="col-md-8">
                            <h4 class="font-weight-bold mb-3">{{ $paymentMethod['name'] }}</h4>
                            
                            <div class="row mb-3">
                                <div class="col-6">
                                    <label class="text-muted">Payment Type</label>
                                    <p class="font-weight-bold">
                                        {{ ucfirst(str_replace('_', ' ', $paymentMethod['code'])) }}
                                    </p>
                                </div>
                                <div class="col-6">
                                    <label class="text-muted">Created On</label>
                                    <p class="font-weight-bold">
                                        {{ date('d M Y, H:i', strtotime($paymentMethod['created_at'])) }}
                                    </p>
                                </div>
                            </div>
                            
                            @if($paymentMethod['account_number'] || $paymentMethod['account_name'])
                                <div class="mb-3">
                                    <label class="text-muted">Account Details</label>
                                    <div class="bg-light p-3 rounded">
                                        @if($paymentMethod['account_number'])
                                            <p class="mb-1">
                                                <strong>Account Number:</strong> 
                                                {{ $paymentMethod['account_number'] }}
                                            </p>
                                        @endif
                                        @if($paymentMethod['account_name'])
                                            <p class="mb-0">
                                                <strong>Account Name:</strong> 
                                                {{ $paymentMethod['account_name'] }}
                                            </p>
                                        @endif
                                    </div>
                                </div>
                            @endif
                            
                            @if($paymentMethod['instructions'])
                                <div class="mb-3">
                                    <label class="text-muted">Payment Instructions</label>
                                    <div class="bg-light p-3 rounded">
                                        {{ $paymentMethod['instructions'] }}
                                    </div>
                                </div>
                            @endif
                        </div>
                        
                        <div class="col-md-4">
                            <!-- Statistics Card -->
                            <div class="card bg-light">
                                <div class="card-body">
                                    <h6 class="font-weight-bold mb-3">Performance Statistics</h6>
                                    
                                    <div class="text-center mb-4">
                                        <div class="display-4 font-weight-bold text-primary">
                                            {{ $successRate ?? 100 }}%
                                        </div>
                                        <div class="text-muted">Success Rate</div>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="text-muted">Total Transactions</label>
                                        <p class="font-weight-bold text-info">
                                            {{ count($recentPayments) }} (Recent)
                                        </p>
                                    </div>
                                    
                                    <div class="mb-3">
                                        <label class="text-muted">Average Transaction</label>
                                        <p class="font-weight-bold text-success">
                                            Rp {{ number_format($averageAmount ?? 0, 0, ',', '.') }}
                                        </p>
                                    </div>
                                    
                                    <div class="mb-0">
                                        <label class="text-muted">Total Revenue</label>
                                        <p class="font-weight-bold text-success">
                                            Rp {{ number_format($totalRevenue ?? 0, 0, ',', '.') }}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Recent Payments -->
            <div class="card mt-4">
                <div class="card-header">
                    <h6 class="card-title mb-0">Recent Payments</h6>
                </div>
                <div class="card-body">
                    @if(count($recentPayments) > 0)
                        <div class="table-responsive">
                            <table class="table table-sm">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Payment ID</th>
                                        <th>Order ID</th>
                                        <th>Customer</th>
                                        <th>Amount</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    @foreach($recentPayments as $payment)
                                    <tr>
                                        <td>{{ date('d M Y, H:i', strtotime($payment['created_at'])) }}</td>
                                        <td>#{{ $payment['id'] }}</td>
                                        <td>
                                            <a href="{{ route('employee.orders.show', $payment['order_id']) }}" target="_blank">
                                                #{{ $payment['order_id'] }}
                                            </a>
                                        </td>
                                        <td>{{ $payment['customer_name'] ?? 'N/A' }}</td>
                                        <td>Rp {{ number_format($payment['amount'], 0, ',', '.') }}</td>
                                        <td>
                                            <span class="badge bg-{{ $payment['status'] == 'completed' ? 'success' : 
                                                                    ($payment['status'] == 'pending' ? 'warning' : 
                                                                    ($payment['status'] == 'failed' ? 'danger' : 'secondary')) }}">
                                                {{ ucfirst($payment['status']) }}
                                            </span>
                                        </td>
                                    </tr>
                                    @endforeach
                                </tbody>
                            </table>
                        </div>
                    @else
                        <p class="text-muted text-center mb-0">No recent payments recorded</p>
                    @endif
                </div>
            </div>
        </div>
        
        <div class="col-lg-4">
            <!-- Actions Card -->
            <div class="card">
                <div class="card-header">
                    <h6 class="card-title mb-0">Actions</h6>
                </div>
                <div class="card-body">
                    <div class="d-grid gap-2">
                        <a href="{{ route('admin.payment-methods.edit', $paymentMethod['id']) }}" class="btn btn-warning btn-block">
                            <i class="fas fa-edit mr-1"></i> Edit Payment Method
                        </a>
                        
                        <form action="{{ route('admin.payment-methods.destroy', $paymentMethod['id']) }}" method="POST" 
                              onsubmit="return confirm('Are you sure you want to delete this payment method?');" class="d-grid">
                            @csrf
                            @method('DELETE')
                            <button type="submit" class="btn btn-danger btn-block">
                                <i class="fas fa-trash mr-1"></i> Delete Payment Method
                            </button>
                        </form>
                        
                        @if($paymentMethod['account_number'])
                            <button type="button" class="btn btn-info btn-block" onclick="copyAccountDetails()">
                                <i class="fas fa-copy mr-1"></i> Copy Account Details
                            </button>
                        @endif
                        
                        <a href="{{ route('admin.payment-methods.index') }}" class="btn btn-secondary btn-block">
                            <i class="fas fa-arrow-left mr-1"></i> Back to List
                        </a>
                    </div>
                </div>
            </div>
            
            <!-- Usage Chart -->
            <div class="card mt-4">
                <div class="card-header">
                    <h6 class="card-title mb-0">Usage Distribution</h6>
                </div>
                <div class="card-body">
                    <canvas id="usageChart" height="200"></canvas>
                </div>
            </div>
        </div>
    </div>
</div>
@endsection

@php
function getPaymentMethodColor($code) {
    $colors = [
        'cash' => 'success',
        'bank_transfer' => 'primary',
        'ewallet' => 'info',
        'qris' => 'warning',
        'credit_card' => 'danger'
    ];
    return $colors[$code] ?? 'secondary';
}
@endphp

@section('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
    document.addEventListener('DOMContentLoaded', function() {
        // Usage Chart
        const usageCtx = document.getElementById('usageChart').getContext('2d');
        new Chart(usageCtx, {
            type: 'doughnut',
            data: {
                labels: ['Completed', 'Pending', 'Failed', 'Cancelled'],
                datasets: [{
                    data: [70, 15, 10, 5],
                    backgroundColor: [
                        '#1cc88a',
                        '#f6c23e',
                        '#e74a3b',
                        '#858796'
                    ]
                }]
            },
            options: {
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    });
    
    function copyAccountDetails() {
        const accountNumber = "{{ $paymentMethod['account_number'] }}";
        const accountName = "{{ $paymentMethod['account_name'] }}";
        const text = `Account Number: ${accountNumber}\nAccount Name: ${accountName}`;
        
        navigator.clipboard.writeText(text).then(function() {
            alert('Account details copied to clipboard!');
        }, function(err) {
            console.error('Failed to copy account details: ', err);
        });
    }
</script>
@endsection