<?php
// database/seeders/PaymentsSeeder.php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Payment;
use App\Models\Order;

class PaymentsSeeder extends Seeder
{
    public function run(): void
    {
        // Create payments for orders without payments
        $ordersWithoutPayments = Order::doesntHave('payment')->get();
        
        foreach ($ordersWithoutPayments as $order) {
            $status = $this->determinePaymentStatus($order->status);
            
            Payment::factory()->create([
                'order_id' => $order->id,
                'status' => $status,
                'amount' => $order->final_amount,
                'payment_date' => $status === 'confirmed' ? $order->created_at : null,
                'confirmed_by' => $status === 'confirmed' 
                    ? \App\Models\User::where('role', 'employee')->inRandomOrder()->first()->id
                    : null,
                'confirmed_at' => $status === 'confirmed' 
                    ? $order->created_at->addMinutes(rand(5, 60))
                    : null,
            ]);
        }
        
        // Create some overdue payments (waiting confirmation for more than 24 hours)
        Payment::where('status', 'waiting_confirmation')
            ->where('created_at', '<', now()->subDay())
            ->take(3)
            ->get()
            ->each(function ($payment) {
                $payment->update([
                    'notes' => 'Payment overdue, please follow up',
                ]);
            });
        
        $this->command->info('✅ Payments seeded! Total: ' . Payment::count());
    }
    
    private function determinePaymentStatus($orderStatus): string
    {
        $statusMap = [
            'pending' => 'pending',
            'confirmed' => 'waiting_confirmation',
            'preparing' => 'confirmed',
            'ready' => 'confirmed',
            'completed' => 'confirmed',
            'cancelled' => 'failed',
        ];
        
        return $statusMap[$orderStatus] ?? 'pending';
    }
}