<?php
// database/seeders/OrdersSeeder.php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Order;
use App\Models\User;

class OrdersSeeder extends Seeder
{
    public function run(): void
    {
        // Get customers and employees
        $customers = User::where('role', 'customer')->get();
        $employees = User::where('role', 'employee')->get();
        
        if ($customers->isEmpty() || $employees->isEmpty()) {
            $this->command->warn('⚠️  No customers or employees found. Please run UsersSeeder first!');
            return;
        }
        
        // Create orders for each customer
        foreach ($customers as $customer) {
            // Each customer has 2-5 orders
            $orderCount = rand(2, 5);
            
            for ($i = 0; $i < $orderCount; $i++) {
                $order = Order::factory()->create([
                    'customer_id' => $customer->id,
                    'employee_id' => $employees->random()->id,
                ]);
                
                // Add order items
                $this->createOrderItems($order);
                
                // Create payment for order
                $this->createPayment($order);
            }
        }
        
        // Create some orders without employees (pending)
        Order::factory()
            ->count(5)
            ->state(['employee_id' => null, 'status' => 'pending'])
            ->create()
            ->each(function ($order) {
                $this->createOrderItems($order);
            });
        
        // Create some cancelled orders
        Order::factory()
            ->count(3)
            ->cancelled()
            ->create()
            ->each(function ($order) {
                $this->createOrderItems($order);
                $this->createPayment($order);
            });
        
        // Create some completed orders
        Order::factory()
            ->count(10)
            ->completed()
            ->create()
            ->each(function ($order) {
                $this->createOrderItems($order);
                $this->createPayment($order, 'confirmed');
            });
        
        $this->command->info('✅ Orders seeded! Total: ' . Order::count());
    }
    
    private function createOrderItems($order): void
    {
        $menuCount = rand(1, 4);
        $menus = \App\Models\Menu::inRandomOrder()->take($menuCount)->get();
        
        $totalAmount = 0;
        
        foreach ($menus as $menu) {
            $quantity = rand(1, 3);
            $unitPrice = $menu->price;
            $subtotal = $quantity * $unitPrice;
            
            \App\Models\OrderItem::create([
                'order_id' => $order->id,
                'menu_id' => $menu->id,
                'quantity' => $quantity,
                'unit_price' => $unitPrice,
                'subtotal' => $subtotal,
                'rating' => $order->status === 'completed' ? rand(3, 5) : null,
                'review_comment' => $order->status === 'completed' && rand(0, 1) 
                    ? 'Great food! Will order again.'
                    : null,
                'reviewed_at' => $order->status === 'completed' && rand(0, 1)
                    ? now()->subDays(rand(1, 30))
                    : null,
            ]);
            
            $totalAmount += $subtotal;
        }
        
        // Update order amounts
        $taxAmount = $totalAmount * 0.11;
        $serviceFee = rand(0, 1) ? 5000 : 0;
        $discountAmount = rand(0, 1) ? rand(5000, 20000) : 0;
        $finalAmount = $totalAmount + $taxAmount + $serviceFee - $discountAmount;
        
        $order->update([
            'total_amount' => $totalAmount,
            'tax_amount' => $taxAmount,
            'service_fee' => $serviceFee,
            'discount_amount' => $discountAmount,
            'final_amount' => $finalAmount,
        ]);
    }
    
    private function createPayment($order, $status = null): void
    {
        if (!$status) {
            $statuses = ['pending', 'waiting_confirmation', 'confirmed', 'failed'];
            $status = $order->status === 'completed' ? 'confirmed' : $this->getRandomStatus($order->status);
        }
        
        \App\Models\Payment::create([
            'order_id' => $order->id,
            'payment_method_id' => \App\Models\PaymentMethod::inRandomOrder()->first()->id,
            'amount' => $order->final_amount,
            'status' => $status,
            'proof_image' => $status === 'waiting_confirmation' 
                ? 'payments/proof_' . uniqid() . '.jpg'
                : null,
            'payment_date' => $status === 'confirmed' ? $order->created_at : null,
            'confirmed_by' => $status === 'confirmed' 
                ? \App\Models\User::where('role', 'employee')->inRandomOrder()->first()->id
                : null,
            'confirmed_at' => $status === 'confirmed' 
                ? $order->created_at->addMinutes(rand(5, 60))
                : null,
            'reference_number' => 'REF' . strtoupper(uniqid()),
        ]);
    }
    
    private function getRandomStatus($orderStatus): string
    {
        $statusMap = [
            'pending' => ['pending'],
            'confirmed' => ['pending', 'waiting_confirmation'],
            'preparing' => ['waiting_confirmation', 'confirmed'],
            'ready' => ['confirmed'],
            'completed' => ['confirmed'],
            'cancelled' => ['failed', 'refunded'],
        ];
        
        $possibleStatuses = $statusMap[$orderStatus] ?? ['pending'];
        return $possibleStatuses[array_rand($possibleStatuses)];
    }
}