<?php
// database/seeders/MenusSeeder.php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Menu;
use App\Models\Category;

class MenusSeeder extends Seeder
{
    // Data lengkap semua menu Korea
    private $koreanMenus = [
        // MAIN COURSES (Category ID: 1)
        [
            'name' => 'Bibimbap',
            'description' => 'Nasi campur Korea dengan sayuran, daging, dan saus gochujang',
            'price' => 45000,
            'category_id' => 1,
            'image' => 'bibimbap.jpg',
            'is_best_seller' => true,
            'is_special' => false,
            'stock' => 50
        ],
        [
            'name' => 'Bulgogi',
            'description' => 'Daging sapi marinasi yang dipanggang dengan saus manis',
            'price' => 75000,
            'category_id' => 1,
            'image' => 'bulgogi.jpg',
            'is_best_seller' => false,
            'is_special' => true,
            'stock' => 40
        ],
        [
            'name' => 'Galbi',
            'description' => 'Iga sapi Korea dengan saus marinasi khusus',
            'price' => 95000,
            'category_id' => 1,
            'image' => 'galbi.jpg',
            'is_best_seller' => true,
            'is_special' => true,
            'stock' => 35
        ],
        [
            'name' => 'Samgyeopsal',
            'description' => 'Daging perut babi panggang disajikan dengan sayuran',
            'price' => 85000,
            'category_id' => 1,
            'image' => 'samgyeopsal.jpg',
            'is_best_seller' => false,
            'is_special' => true,
            'stock' => 45
        ],
        [
            'name' => 'Bossam',
            'description' => 'Daging babi rebus dengan kimchi dan saus',
            'price' => 65000,
            'category_id' => 1,
            'image' => 'bossam.jpg',
            'is_best_seller' => false,
            'is_special' => false,
            'stock' => 30
        ],
        
        // SIDE DISHES (Category ID: 2)
        [
            'name' => 'Kimchi',
            'description' => 'Sayuran fermentasi pedas khas Korea',
            'price' => 25000,
            'category_id' => 2,
            'image' => 'kimchi.jpg',
            'is_best_seller' => true,
            'is_special' => false,
            'stock' => 100
        ],
        [
            'name' => 'Tteokbokki',
            'description' => 'Kue beras pedas dengan saus gochujang',
            'price' => 35000,
            'category_id' => 2,
            'image' => 'tteokbokki.jpg',
            'is_best_seller' => true,
            'is_special' => false,
            'stock' => 60
        ],
        [
            'name' => 'Japchae',
            'description' => 'Mi kaca dengan sayuran dan daging sapi',
            'price' => 55000,
            'category_id' => 2,
            'image' => 'japchae.jpg',
            'is_best_seller' => false,
            'is_special' => false,
            'stock' => 40
        ],
        [
            'name' => 'Haemul Pajeon',
            'description' => 'Panekuk seafood dengan daun bawang',
            'price' => 42000,
            'category_id' => 2,
            'image' => 'haemul-pajeon.jpg',
            'is_best_seller' => false,
            'is_special' => true,
            'stock' => 35
        ],
        
        // NOODLES & RICE (Category ID: 3)
        [
            'name' => 'Naengmyeon',
            'description' => 'Mi dingin Korea dengan kaldu es',
            'price' => 48000,
            'category_id' => 3,
            'image' => 'naengmyeon.jpg',
            'is_best_seller' => false,
            'is_special' => false,
            'stock' => 25
        ],
        [
            'name' => 'Jjajangmyeon',
            'description' => 'Mi dengan saus kacang hitam',
            'price' => 40000,
            'category_id' => 3,
            'image' => 'jjajangmyeon.jpg',
            'is_best_seller' => true,
            'is_special' => false,
            'stock' => 50
        ],
        [
            'name' => 'Kimbap',
            'description' => 'Roll nasi Korea dengan berbagai isian',
            'price' => 30000,
            'category_id' => 3,
            'image' => 'kimbap.jpg',
            'is_best_seller' => true,
            'is_special' => false,
            'stock' => 80
        ],
        
        // SOUPS & STEWS (Category ID: 4)
        [
            'name' => 'Sundubu Jjigae',
            'description' => 'Sup tofu pedas dengan seafood',
            'price' => 38000,
            'category_id' => 4,
            'image' => 'sundubu-jjigae.jpg',
            'is_best_seller' => false,
            'is_special' => true,
            'stock' => 40
        ],
        [
            'name' => 'Kimchi Jjigae',
            'description' => 'Sup kimchi dengan daging babi dan tofu',
            'price' => 38000,
            'category_id' => 4,
            'image' => 'kimchi-jjigae.jpg',
            'is_best_seller' => true,
            'is_special' => false,
            'stock' => 45
        ],
        [
            'name' => 'Yukgaejang',
            'description' => 'Sup daging sapi pedas dengan sayuran',
            'price' => 42000,
            'category_id' => 4,
            'image' => 'yukgaejang.jpg',
            'is_best_seller' => false,
            'is_special' => false,
            'stock' => 30
        ],
        
        // DRINKS (Category ID: 5)
        [
            'name' => 'Soju',
            'description' => 'Minuman beralkohol tradisional Korea',
            'price' => 55000,
            'category_id' => 5,
            'image' => 'soju.jpg',
            'is_best_seller' => true,
            'is_special' => false,
            'stock' => 100
        ],
        [
            'name' => 'Makgeolli',
            'description' => 'Minuman beras fermentasi Korea',
            'price' => 45000,
            'category_id' => 5,
            'image' => 'makgeolli.jpg',
            'is_best_seller' => false,
            'is_special' => true,
            'stock' => 60
        ],
        [
            'name' => 'Sikhye',
            'description' => 'Minuman beras manis tradisional',
            'price' => 25000,
            'category_id' => 5,
            'image' => 'sikhye.jpg',
            'is_best_seller' => false,
            'is_special' => false,
            'stock' => 40
        ],
    ];

    public function run(): void
    {
        // Pastikan semua kategori ada
        $this->ensureCategoriesExist();
        
        $createdCount = 0;
        $updatedCount = 0;
        
        foreach ($this->koreanMenus as $menuData) {
            // Cek apakah menu sudah ada
            $existingMenu = Menu::where('name', $menuData['name'])
                ->where('category_id', $menuData['category_id'])
                ->first();
            
            if ($existingMenu) {
                // Update menu yang sudah ada
                $existingMenu->update([
                    'description' => $menuData['description'],
                    'price' => $menuData['price'],
                    'image_url' => "images/menu/{$menuData['image']}",
                    'stock' => $menuData['stock'],
                    'is_available' => $menuData['stock'] > 0,
                    'is_special' => $menuData['is_special'],
                    'is_best_seller' => $menuData['is_best_seller'],
                ]);
                $updatedCount++;
            } else {
                // Buat menu baru
                Menu::create([
                    'category_id' => $menuData['category_id'],
                    'name' => $menuData['name'],
                    'description' => $menuData['description'],
                    'price' => $menuData['price'],
                    'image_url' => "images/menu/{$menuData['image']}",
                    'stock' => $menuData['stock'],
                    'is_available' => $menuData['stock'] > 0,
                    'is_special' => $menuData['is_special'],
                    'is_best_seller' => $menuData['is_best_seller'],
                ]);
                $createdCount++;
            }
        }
        
        // HAPUS menu yang dibuat oleh factory sebelumnya
        // Cari menu yang tidak ada di list kita tapi ada di database
        $ourMenuNames = array_column($this->koreanMenus, 'name');
        
        // Option 1: Soft delete (jika ada soft deletes)
        // Menu::whereNotIn('name', $ourMenuNames)->delete();
        
        // Option 2: Set is_available = false
        Menu::whereNotIn('name', $ourMenuNames)
            ->update([
                'is_available' => false,
                'stock' => 0
            ]);
            
        $hiddenCount = Menu::whereNotIn('name', $ourMenuNames)->count();
        
        $this->command->info("✅ Menu seeding completed!");
        $this->command->info("   Created: {$createdCount} new menus");
        $this->command->info("   Updated: {$updatedCount} existing menus");
        $this->command->info("   Hidden: {$hiddenCount} old factory menus");
        $this->command->info("   Total active: " . Menu::where('is_available', true)->count());
    }
    
    private function ensureCategoriesExist(): void
    {
        $categories = [
            ['name' => 'Main Course', 'description' => 'Main dishes', 'is_active' => true],
            ['name' => 'Side Dishes', 'description' => 'Korean side dishes', 'is_active' => true],
            ['name' => 'Noodles & Rice', 'description' => 'Noodle and rice dishes', 'is_active' => true],
            ['name' => 'Soups & Stews', 'description' => 'Korean soups and stews', 'is_active' => true],
            ['name' => 'Drinks', 'description' => 'Beverages', 'is_active' => true],
        ];
        
        foreach ($categories as $categoryData) {
            Category::firstOrCreate(
                ['name' => $categoryData['name']],
                $categoryData
            );
        }
    }
}