<?php
// database/seeders/JobApplicationsSeeder.php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\JobApplication;
use App\Models\JobPosting;
use App\Models\User;

class JobApplicationsSeeder extends Seeder
{
    public function run(): void
    {
        $jobPostings = JobPosting::where('is_active', true)->get();
        $customers = User::where('role', 'customer')->get();
        
        if ($jobPostings->isEmpty() || $customers->isEmpty()) {
            $this->command->warn('⚠️  No active job postings or customers found.');
            return;
        }
        
        // Create applications for each job posting
        foreach ($jobPostings as $job) {
            $applicationCount = rand(3, 8);
            
            for ($i = 0; $i < $applicationCount; $i++) {
                $customer = $customers->random();
                
                JobApplication::create([
                    'job_posting_id' => $job->id,
                    'applicant_name' => $customer->name,
                    'applicant_email' => $customer->email,
                    'applicant_phone' => $customer->phone,
                    'applicant_cv' => 'cvs/' . str_replace(' ', '_', strtolower($customer->name)) . '_cv.pdf',
                    'cover_letter' => $this->generateCoverLetter($customer->name, $job->title),
                    'status' => $this->getRandomStatus(),
                    'notes' => rand(0, 1) ? 'Candidate looks promising' : null,
                    'submitted_at' => now()->subDays(rand(1, 30)),
                    'created_at' => now()->subDays(rand(1, 30)),
                    'updated_at' => now(),
                ]);
            }
        }
        
        // Create some applications from external applicants
        JobApplication::factory()->count(15)->create();
        
        // Update job posting views count based on applications
        foreach ($jobPostings as $job) {
            $applicationCount = JobApplication::where('job_posting_id', $job->id)->count();
            $job->update([
                'views_count' => $applicationCount * rand(5, 10),
            ]);
        }
        
        $this->command->info('✅ Job applications seeded! Total: ' . JobApplication::count());
    }
    
    private function generateCoverLetter($name, $jobTitle): string
    {
        return "Dear Hiring Manager,\n\n" .
               "I am writing to apply for the $jobTitle position at Chingu Bite. " .
               "I am excited about the opportunity to contribute to your team.\n\n" .
               "Sincerely,\n$name";
    }
    
    private function getRandomStatus(): string
    {
        $statuses = ['pending', 'reviewed', 'interviewed', 'accepted', 'rejected'];
        $weights = [40, 25, 20, 10, 5]; // Probability weights
        
        $random = rand(1, 100);
        $cumulative = 0;
        
        for ($i = 0; $i < count($statuses); $i++) {
            $cumulative += $weights[$i];
            if ($random <= $cumulative) {
                return $statuses[$i];
            }
        }
        
        return 'pending';
    }
}