<?php
// database/seeders/CartsSeeder.php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use App\Models\Cart;
use App\Models\User;
use App\Models\Menu;

class CartsSeeder extends Seeder
{
    public function run(): void
    {
        // Hanya seed jika tabel carts ada
        if (!\Schema::hasTable('carts')) {
            $this->command->warn('⚠️  Carts table does not exist. Skipping...');
            return;
        }
        
        $customers = User::where('role', 'customer')->get();
        $menus = Menu::where('is_available', true)->get();
        
        if ($customers->isEmpty() || $menus->isEmpty()) {
            $this->command->warn('⚠️  No customers or menus found for carts.');
            return;
        }
        
        // Create active carts for some customers
        foreach ($customers->take(10) as $customer) {
            $cartExists = Cart::where('user_id', $customer->id)
                ->where('is_active', true)
                ->exists();
            
            if (!$cartExists && rand(0, 1)) {
                $cart = Cart::create([
                    'user_id' => $customer->id,
                    'is_active' => true,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
                
                // Add 1-3 items to cart
                $menuCount = rand(1, 3);
                $selectedMenus = $menus->random($menuCount);
                
                foreach ($selectedMenus as $menu) {
                    $quantity = rand(1, 3);
                    
                    // Check if item already in cart
                    $existingItem = \DB::table('cart_items')
                        ->where('cart_id', $cart->id)
                        ->where('menu_id', $menu->id)
                        ->first();
                    
                    if (!$existingItem) {
                        \DB::table('cart_items')->insert([
                            'cart_id' => $cart->id,
                            'menu_id' => $menu->id,
                            'quantity' => $quantity,
                            'unit_price' => $menu->price,
                            'subtotal' => $quantity * $menu->price,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                }
                
                // Update cart total
                $total = \DB::table('cart_items')
                    ->where('cart_id', $cart->id)
                    ->sum('subtotal');
                
                $cart->update(['total_amount' => $total]);
            }
        }
        
        // Create some abandoned carts (not active)
        foreach ($customers->take(5) as $customer) {
            if (!Cart::where('user_id', $customer->id)->exists()) {
                Cart::create([
                    'user_id' => $customer->id,
                    'is_active' => false,
                    'total_amount' => 0,
                    'created_at' => now()->subDays(rand(1, 30)),
                    'updated_at' => now()->subDays(rand(1, 30)),
                ]);
            }
        }
        
        $totalCarts = Cart::count();
        $activeCarts = Cart::where('is_active', true)->count();
        
        $this->command->info("✅ Carts seeded! Total: {$totalCarts}, Active: {$activeCarts}");
    }
}