<?php
// database/factories/OrderFactory.php

namespace Database\Factories;

use Illuminate\Database\Eloquent\Factories\Factory;

class OrderFactory extends Factory
{
    public function definition(): array
    {
        $orderTypes = ['dine_in', 'takeaway', 'delivery'];
        $statuses = ['pending', 'confirmed', 'preparing', 'ready', 'completed', 'cancelled'];
        
        $customer = \App\Models\User::where('role', 'customer')->inRandomOrder()->first();
        
        // Calculate amounts
        $totalAmount = $this->faker->numberBetween(50000, 500000);
        $taxAmount = $totalAmount * 0.11; // 11% tax
        $serviceFee = $this->faker->optional(0.7, 0)->numberBetween(5000, 20000);
        $discountAmount = $this->faker->randomElement([0, 5000, 20000]);        
        $finalAmount = $totalAmount + $taxAmount + $serviceFee - $discountAmount;
        
        static $orderNumberCounter = 1000;
        $orderNumberCounter++;
        
        return [
            'order_number' => 'ORD' . date('Ymd') . str_pad($orderNumberCounter, 4, '0', STR_PAD_LEFT),
            'customer_id' => $customer->id ?? \App\Models\User::factory()->customer(),
            'employee_id' => \App\Models\User::where('role', 'employee')->inRandomOrder()->first()->id ?? \App\Models\User::factory()->employee(),
            'table_number' => $this->faker->numberBetween(1, 20),
            'order_type' => $this->faker->randomElement($orderTypes),
            'status' => $this->faker->randomElement($statuses),
            'total_amount' => $totalAmount,
            'tax_amount' => $taxAmount,
            'service_fee' => $serviceFee,
            'discount_amount' => $discountAmount,
            'final_amount' => $finalAmount,
            'notes' => $this->faker->sentence(),
            'scheduled_for' => $this->faker->optional(0.1)->dateTimeBetween('+1 hour', '+1 day'),
            'completed_at' => function (array $attributes) {
                return $attributes['status'] === 'completed' 
                    ? $this->faker->dateTimeBetween('-6 months', 'now')
                    : null;
            },
            'created_at' => $this->faker->dateTimeBetween('-6 months', 'now'),
            'updated_at' => now(),
        ];
    }
    
    public function dineIn(): static
    {
        return $this->state(fn (array $attributes) => [
            'order_type' => 'dine_in',
            'table_number' => $this->faker->numberBetween(1, 20),
        ]);
    }
    
    public function takeaway(): static
    {
        return $this->state(fn (array $attributes) => [
            'order_type' => 'takeaway',
            'table_number' => null,
        ]);
    }
    
    public function delivery(): static
    {
        return $this->state(fn (array $attributes) => [
            'order_type' => 'delivery',
            'table_number' => null,
        ]);
    }
    
    public function completed(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'completed',
            'completed_at' => $this->faker->dateTimeBetween('-1 month', 'now'),
        ]);
    }
    
    public function cancelled(): static
    {
        return $this->state(fn (array $attributes) => [
            'status' => 'cancelled',
        ]);
    }
}