<?php
// app/Services/OrderService.php
namespace App\Services;

use App\Models\Promo;
use App\Models\Menu;

class OrderService
{
    /**
     * Calculate order totals
     */
    public function calculateOrder(array $cart, array $checkoutData): array
    {
        $subtotal = 0;
        $items = [];
        $availableItems = [];

        // Process cart items
        foreach ($cart as $cartItem) {
            $menu = Menu::available()->find($cartItem['menu_id']);
            
            if (!$menu || $menu->stock < $cartItem['quantity']) {
                continue; // Skip unavailable items
            }

            $itemTotal = $menu->price * $cartItem['quantity'];
            $subtotal += $itemTotal;

            $items[] = [
                'menu_id' => $menu->id,
                'menu' => $menu,
                'quantity' => $cartItem['quantity'],
                'unit_price' => $menu->price,
                'subtotal' => $itemTotal,
                'notes' => $cartItem['notes'] ?? null,
            ];
            
            $availableItems[] = $menu->id;
        }

        // Get system settings (default values)
        $taxRate = 10; // 10%
        $serviceFee = 5000; // Rp 5,000

        // Calculate tax and service fee
        $tax = ($subtotal * $taxRate) / 100;
        $serviceFeeAmount = $serviceFee;

        // Apply promo if available
        $discount = 0;
        if (!empty($checkoutData['promo_code'])) {
            $promo = Promo::where('code', $checkoutData['promo_code'])->first();
            if ($promo) {
                $discount = $promo->calculateDiscount($subtotal);
            }
        }

        // Calculate final total
        $finalTotal = $subtotal + $tax + $serviceFeeAmount - $discount;

        return [
            'subtotal' => $subtotal,
            'tax' => $tax,
            'service_fee' => $serviceFeeAmount,
            'discount' => $discount,
            'final_total' => $finalTotal,
            'total' => $finalTotal, // alias
            'items' => $items,
            'available_items' => $availableItems,
        ];
    }

    /**
     * Check if cart can be checked out
     */
    public function validateCart(array $cart): array
    {
        $total = 0;
        $itemCount = 0;
        $availableItems = 0;
        $unavailableItems = [];
        $items = [];

        foreach ($cart as $cartItem) {
            $menu = Menu::find($cartItem['menu_id']);
            
            if (!$menu) {
                $unavailableItems[] = [
                    'menu_id' => $cartItem['menu_id'],
                    'reason' => 'Menu tidak ditemukan'
                ];
                continue;
            }

            $itemCount++;
            $isAvailable = $menu->is_available && $menu->stock >= $cartItem['quantity'];
            
            if ($isAvailable) {
                $availableItems++;
                $subtotal = $menu->price * $cartItem['quantity'];
                $total += $subtotal;
                
                $items[] = [
                    'menu_id' => $menu->id,
                    'name' => $menu->name,
                    'price' => $menu->price,
                    'quantity' => $cartItem['quantity'],
                    'subtotal' => $subtotal,
                    'notes' => $cartItem['notes'] ?? null,
                    'is_available' => true,
                ];
            } else {
                $unavailableItems[] = [
                    'menu_id' => $menu->id,
                    'name' => $menu->name,
                    'reason' => $menu->is_available ? 
                        "Stok tidak mencukupi (tersedia: {$menu->stock})" : 
                        "Menu tidak tersedia"
                ];
            }
        }

        // Get system settings
        $taxRate = 10;
        $serviceFee = 5000;

        $tax = ($total * $taxRate) / 100;
        $finalTotal = $total + $tax + $serviceFee;

        return [
            'can_checkout' => $availableItems > 0 && count($unavailableItems) === 0,
            'item_count' => $itemCount,
            'available_items' => $availableItems,
            'unavailable_items' => $unavailableItems,
            'subtotal' => $total,
            'tax_amount' => $tax,
            'service_fee' => $serviceFee,
            'estimated_total' => $finalTotal,
            'items' => $items,
            'summary' => [
                'formatted_subtotal' => 'Rp ' . number_format($total, 0, ',', '.'),
                'formatted_tax' => 'Rp ' . number_format($tax, 0, ',', '.'),
                'formatted_service_fee' => 'Rp ' . number_format($serviceFee, 0, ',', '.'),
                'formatted_total' => 'Rp ' . number_format($finalTotal, 0, ',', '.'),
            ]
        ];
    }

    /**
     * Generate order number
     */
    public function generateOrderNumber(): string
    {
        $date = date('Ymd');
        $lastOrder = \App\Models\Order::whereDate('created_at', today())->latest()->first();
        
        if ($lastOrder) {
            // Extract sequence from existing order number
            $parts = explode('-', $lastOrder->order_number);
            $lastSequence = (int) end($parts);
            $sequence = $lastSequence + 1;
        } else {
            $sequence = 1;
        }

        return 'ORD-' . $date . '-' . str_pad($sequence, 6, '0', STR_PAD_LEFT);
    }
}