<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

/**
 * @property int $id
 * @property int $order_id
 * @property int $payment_method_id
 * @property numeric $amount
 * @property string $status
 * @property string|null $proof_image
 * @property \Illuminate\Support\Carbon|null $payment_date
 * @property int|null $confirmed_by
 * @property \Illuminate\Support\Carbon|null $confirmed_at
 * @property string|null $reference_number
 * @property string|null $notes
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property-read \App\Models\User|null $confirmedBy
 * @property-read \App\Models\Order $order
 * @property-read \App\Models\PaymentMethod $paymentMethod
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment confirmed()
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment newModelQuery()
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment newQuery()
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment pending()
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment query()
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment waitingConfirmation()
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereAmount($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereConfirmedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereConfirmedBy($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereCreatedAt($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereId($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereNotes($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereOrderId($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment wherePaymentDate($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment wherePaymentMethodId($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereProofImage($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereReferenceNumber($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereStatus($value)
 * @method static \Illuminate\Database\Eloquent\Builder<static>|Payment whereUpdatedAt($value)
 * @mixin \Eloquent
 */
class Payment extends Model
{
    use HasFactory;

    protected $fillable = [
        'order_id',
        'payment_method_id',
        'amount',
        'status',
        'proof_image',
        'payment_date',
        'confirmed_by',
        'confirmed_at',
        'reference_number',
        'notes',
    ];

    protected $casts = [
        'status' => 'string',
        'amount' => 'decimal:2',
        'payment_date' => 'datetime',
        'confirmed_at' => 'datetime',
    ];

    // Relationships
    public function order()
    {
        return $this->belongsTo(Order::class);
    }

    public function paymentMethod()
    {
        return $this->belongsTo(PaymentMethod::class);
    }

    public function confirmedBy()
    {
        return $this->belongsTo(User::class, 'confirmed_by');
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeWaitingConfirmation($query)
    {
        return $query->where('status', 'waiting_confirmation');
    }

    public function scopeConfirmed($query)
    {
        return $query->where('status', 'confirmed');
    }

    // Helpers
    public function confirm($confirmedBy, $notes = null): bool
    {
        $this->status = 'confirmed';
        $this->confirmed_by = $confirmedBy;
        $this->confirmed_at = now();
        $this->notes = $notes;
        return $this->save();
    }

    public function reject($notes = null): bool
    {
        $this->status = 'failed';
        $this->notes = $notes;
        return $this->save();
    }

    public function isConfirmed(): bool
    {
        return $this->status === 'confirmed';
    }

        public function canUploadProof(): bool
    {
        return in_array($this->status, ['pending', 'waiting_confirmation']);
    }

    public function getStatusBadge(): string
    {
        $badges = [
            'pending' => 'warning',
            'waiting_confirmation' => 'info',
            'confirmed' => 'success',
            'failed' => 'danger',
            'refunded' => 'secondary',
        ];

        return $badges[$this->status] ?? 'secondary';
    }

    public function getFormattedStatus(): string
    {
        $statuses = [
            'pending' => 'Menunggu Pembayaran',
            'waiting_confirmation' => 'Menunggu Konfirmasi',
            'confirmed' => 'Terkonfirmasi',
            'failed' => 'Gagal',
            'refunded' => 'Dikembalikan',
        ];

        return $statuses[$this->status] ?? $this->status;
    }

    public function isOverdue(): bool
    {
        // Payment confirmation timeout (24 hours from creation)
        $timeoutHours = 24;
        return $this->status === 'waiting_confirmation' && 
               $this->created_at->addHours($timeoutHours)->isPast();
    }
}
