<?php
namespace App\Http\Resources\Api\Employee;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class PaymentResource extends JsonResource
{
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'order_id' => $this->order_id,
            'order_number' => $this->order->order_number ?? null,
            'amount' => (float) $this->amount,
            'formatted_amount' => 'Rp ' . number_format($this->amount, 0, ',', '.'),
            'status' => $this->status,
            'formatted_status' => $this->getFormattedStatus(),
            'status_badge' => $this->getStatusBadge(),
            'is_overdue' => $this->isOverdue(),
            'waiting_time' => $this->getWaitingTime(),
            'proof_image' => $this->proof_image ? url('storage/' . $this->proof_image) : null,
            'proof_filename' => $this->proof_image ? basename($this->proof_image) : null,
            'payment_date' => $this->payment_date?->format('Y-m-d H:i:s'),
            'reference_number' => $this->reference_number,
            'notes' => $this->notes,
            'confirmed_at' => $this->confirmed_at?->format('Y-m-d H:i:s'),
            'confirmed_by' => $this->confirmedBy ? [
                'id' => $this->confirmedBy->id,
                'name' => $this->confirmedBy->name,
            ] : null,
            'payment_method' => $this->paymentMethod ? [
                'id' => $this->paymentMethod->id,
                'name' => $this->paymentMethod->name,
                'code' => $this->paymentMethod->code,
            ] : null,
            'customer' => $this->order->customer ? [
                'id' => $this->order->customer->id,
                'name' => $this->order->customer->name,
                'phone' => $this->order->customer->phone,
                'email' => $this->order->customer->email,
            ] : null,
            'created_at' => $this->created_at->format('Y-m-d H:i:s'),
            'time_ago' => $this->created_at->diffForHumans(),
            'can_confirm' => $this->status === 'waiting_confirmation',
            'can_reject' => $this->status === 'waiting_confirmation',
            'actions' => $this->getAvailableActions(),
        ];
    }

    private function getFormattedStatus(): string
    {
        $statuses = [
            'pending' => 'Menunggu Pembayaran',
            'waiting_confirmation' => 'Menunggu Konfirmasi',
            'confirmed' => 'Terkonfirmasi',
            'failed' => 'Gagal',
            'refunded' => 'Dikembalikan',
        ];

        return $statuses[$this->status] ?? $this->status;
    }

    private function getStatusBadge(): string
    {
        $badges = [
            'pending' => 'warning',
            'waiting_confirmation' => 'info',
            'confirmed' => 'success',
            'failed' => 'danger',
            'refunded' => 'secondary',
        ];

        return $badges[$this->status] ?? 'secondary';
    }

    private function isOverdue(): bool
    {
        return $this->status === 'waiting_confirmation' && 
               $this->created_at->addHours(24)->isPast();
    }

    private function getWaitingTime(): string
    {
        if ($this->status !== 'waiting_confirmation') {
            return 'N/A';
        }

        $minutes = $this->created_at->diffInMinutes(now());
        
        if ($minutes < 60) {
            return $minutes . ' menit';
        } else {
            $hours = floor($minutes / 60);
            $remainingMinutes = $minutes % 60;
            return $hours . ' jam ' . $remainingMinutes . ' menit';
        }
    }

    private function getAvailableActions(): array
    {
        $actions = [];

        if ($this->status === 'waiting_confirmation') {
            $actions[] = [
                'action' => 'confirm',
                'label' => 'Konfirmasi',
                'description' => 'Konfirmasi pembayaran valid',
                'icon' => '✅',
                'method' => 'POST',
                'url' => "/api/employee/payments/{$this->id}/confirm",
            ];
            $actions[] = [
                'action' => 'reject',
                'label' => 'Tolak',
                'description' => 'Tolak pembayaran',
                'icon' => '❌',
                'method' => 'POST',
                'url' => "/api/employee/payments/{$this->id}/reject",
            ];
        }

        return $actions;
    }
}