<?php
// app/Http/Resources/Api/Employee/OrderResource.php
namespace App\Http\Resources\Api\Employee;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class OrderResource extends JsonResource
{
    public function toArray(Request $request): array
    {
        return [
            'id' => $this->id,
            'order_number' => $this->order_number,
            'order_type' => $this->order_type,
            'formatted_order_type' => $this->getFormattedOrderType(),
            'status' => $this->status,
            'formatted_status' => $this->getFormattedStatus(),
            'status_badge' => $this->getStatusBadge(),
            'can_update_status' => $this->canUpdateStatus(),
            'available_statuses' => $this->getAvailableStatuses(),
            'total_amount' => (float) $this->total_amount,
            'tax_amount' => (float) $this->tax_amount,
            'service_fee' => (float) $this->service_fee,
            'discount_amount' => (float) $this->discount_amount,
            'final_amount' => (float) $this->final_amount,
            'formatted_final_amount' => 'Rp ' . number_format($this->final_amount, 0, ',', '.'),
            'table_number' => $this->table_number,
            'notes' => $this->notes,
            'scheduled_for' => $this->scheduled_for?->format('Y-m-d H:i:s'),
            'created_at' => $this->created_at->format('Y-m-d H:i:s'),
            'time_ago' => $this->created_at->diffForHumans(),
            'customer' => $this->whenLoaded('customer', function () {
                return [
                    'id' => $this->customer->id,
                    'name' => $this->customer->name,
                    'phone' => $this->customer->phone,
                    'email' => $this->customer->email,
                ];
            }),
            'employee' => $this->whenLoaded('employee', function () {
                return $this->employee ? [
                    'id' => $this->employee->id,
                    'name' => $this->employee->name,
                ] : null;
            }),
            'items' => $this->whenLoaded('items', function () {
                return $this->items->map(function ($item) {
                    return [
                        'id' => $item->id,
                        'menu_id' => $item->menu_id,
                        'menu_name' => $item->menu->name ?? null,
                        'menu_price' => (float) $item->menu->price ?? 0,
                        'quantity' => $item->quantity,
                        'unit_price' => (float) $item->unit_price,
                        'subtotal' => (float) $item->subtotal,
                        'notes' => $item->notes,
                        'rating' => $item->rating,
                    ];
                });
            }),
            'payments' => $this->whenLoaded('payments', function () {
                return $this->payments->map(function ($payment) {
                    return [
                        'id' => $payment->id,
                        'status' => $payment->status,
                        'formatted_status' => $payment->getFormattedStatus(),
                        'amount' => (float) $payment->amount,
                        'payment_method' => $payment->paymentMethod->name ?? null,
                        'proof_image' => $payment->proof_image ? url('storage/' . $payment->proof_image) : null,
                        'reference_number' => $payment->reference_number,
                        'can_confirm' => $payment->status === 'waiting_confirmation',
                    ];
                });
            }),
            'timeline' => $this->getTimeline(),
        ];
    }

    private function getFormattedOrderType(): string
    {
        return match ($this->order_type) {
            'dine_in' => 'Makan di Tempat',
            'takeaway' => 'Bawa Pulang',
            'delivery' => 'Delivery',
            default => $this->order_type,
        };
    }

    private function getFormattedStatus(): string
    {
        return match ($this->status) {
            'pending' => 'Menunggu Konfirmasi',
            'confirmed' => 'Dikonfirmasi',
            'preparing' => 'Sedang Disiapkan',
            'ready' => 'Siap Diambil',
            'completed' => 'Selesai',
            'cancelled' => 'Dibatalkan',
            default => $this->status,
        };
    }

    private function getStatusBadge(): string
    {
        $badges = [
            'pending' => 'warning',
            'confirmed' => 'info',
            'preparing' => 'primary',
            'ready' => 'success',
            'completed' => 'dark',
            'cancelled' => 'danger',
        ];

        return $badges[$this->status] ?? 'secondary';
    }

    private function canUpdateStatus(): bool
    {
        return !in_array($this->status, ['completed', 'cancelled']);
    }

    private function getAvailableStatuses(): array
    {
        $statuses = [
            'pending' => ['confirmed', 'cancelled'],
            'confirmed' => ['preparing', 'cancelled'],
            'preparing' => ['ready', 'cancelled'],
            'ready' => ['completed'],
            'completed' => [],
            'cancelled' => [],
        ];

        return array_map(function ($status) {
            return [
                'value' => $status,
                'label' => $this->getStatusLabel($status),
            ];
        }, $statuses[$this->status] ?? []);
    }

    private function getStatusLabel($status): string
    {
        $labels = [
            'pending' => 'Menunggu Konfirmasi',
            'confirmed' => 'Konfirmasi',
            'preparing' => 'Mulai Siapkan',
            'ready' => 'Tandai Siap',
            'completed' => 'Selesaikan',
            'cancelled' => 'Batalkan',
        ];

        return $labels[$status] ?? $status;
    }

    private function getTimeline(): array
    {
        $timeline = [];

        // Order created
        $timeline[] = [
            'status' => 'created',
            'title' => 'Pesanan Dibuat',
            'description' => 'Pesanan diterima sistem',
            'time' => $this->created_at->format('H:i'),
            'icon' => '📝',
            'completed' => true,
        ];

        // Status updates
        $statusUpdates = [
            'confirmed' => ['title' => 'Dikonfirmasi', 'icon' => '✅'],
            'preparing' => ['title' => 'Disiapkan', 'icon' => '👨‍🍳'],
            'ready' => ['title' => 'Siap', 'icon' => '📦'],
            'completed' => ['title' => 'Selesai', 'icon' => '🎉'],
            'cancelled' => ['title' => 'Dibatalkan', 'icon' => '❌'],
        ];

        foreach ($statusUpdates as $status => $info) {
            if ($this->status === $status) {
                $timeline[] = [
                    'status' => $status,
                    'title' => $info['title'],
                    'description' => 'Status saat ini',
                    'time' => 'Sekarang',
                    'icon' => $info['icon'],
                    'completed' => false,
                    'current' => true,
                ];
            }
        }

        return $timeline;
    }
}