<?php

namespace App\Http\Resources\Api\Customer;
use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class PaymentResource extends JsonResource
{
    public function toArray($request): array
    {
        return [
            'id' => $this->id,
            'order_id' => $this->order_id,
            'order_number' => $this->order->order_number ?? null,
            'amount' => (float) $this->amount,
            'formatted_amount' => 'Rp ' . number_format($this->amount, 0, ',', '.'),
            'status' => $this->status,
            'formatted_status' => $this->getFormattedStatus(),
            'status_badge' => $this->getStatusBadge(),
            'can_upload_proof' => $this->canUploadProof(),
            'is_overdue' => $this->isOverdue(),
            'proof_image' => $this->proof_image ? url('storage/' . $this->proof_image) : null,
            'proof_image_filename' => $this->proof_image ? basename($this->proof_image) : null,
            'payment_date' => $this->payment_date?->format('Y-m-d H:i:s'),
            'reference_number' => $this->reference_number,
            'notes' => $this->notes,
            'confirmed_at' => $this->confirmed_at?->format('Y-m-d H:i:s'),
            'confirmed_by' => $this->confirmedBy ? [
                'id' => $this->confirmedBy->id,
                'name' => $this->confirmedBy->name,
            ] : null,
            'payment_method' => $this->paymentMethod ? [
                'id' => $this->paymentMethod->id,
                'name' => $this->paymentMethod->name,
                'code' => $this->paymentMethod->code,
                'account_number' => $this->paymentMethod->account_number,
                'account_name' => $this->paymentMethod->account_name,
            ] : null,
            'created_at' => $this->created_at->format('Y-m-d H:i:s'),
            'updated_at' => $this->updated_at->format('Y-m-d H:i:s'),
            'timeline' => $this->getTimeline(),
        ];
    }

        private function getTimeline(): array
    {
        $timeline = [];

        // Payment created
        $timeline[] = [
            'status' => 'created',
            'title' => 'Pembayaran dibuat',
            'description' => 'Tagihan pembayaran telah dibuat',
            'time' => $this->created_at->format('Y-m-d H:i:s'),
            'icon' => '💳',
            'completed' => true,
        ];

        // If proof uploaded
        if ($this->proof_image) {
            $timeline[] = [
                'status' => 'proof_uploaded',
                'title' => 'Bukti diupload',
                'description' => 'Bukti pembayaran telah diupload',
                'time' => $this->updated_at->format('Y-m-d H:i:s'),
                'icon' => '📎',
                'completed' => true,
            ];
        }

        // If waiting confirmation
        if ($this->status === 'waiting_confirmation') {
            $timeline[] = [
                'status' => 'waiting_confirmation',
                'title' => 'Menunggu Konfirmasi',
                'description' => 'Bukti pembayaran sedang diverifikasi',
                'time' => 'Sekarang',
                'icon' => '⏳',
                'completed' => false,
                'current' => true,
            ];
        }

        // If confirmed
        if ($this->status === 'confirmed') {
            $timeline[] = [
                'status' => 'confirmed',
                'title' => 'Terkonfirmasi',
                'description' => 'Pembayaran telah dikonfirmasi',
                'time' => $this->confirmed_at?->format('Y-m-d H:i:s') ?? $this->updated_at->format('Y-m-d H:i:s'),
                'icon' => '✅',
                'completed' => true,
            ];
        }

        // If failed
        if ($this->status === 'failed') {
            $timeline[] = [
                'status' => 'failed',
                'title' => 'Gagal',
                'description' => 'Pembayaran gagal diverifikasi',
                'time' => $this->updated_at->format('Y-m-d H:i:s'),
                'icon' => '❌',
                'completed' => true,
            ];
        }

        // If refunded
        if ($this->status === 'refunded') {
            $timeline[] = [
                'status' => 'refunded',
                'title' => 'Dikembalikan',
                'description' => 'Pembayaran telah dikembalikan',
                'time' => $this->updated_at->format('Y-m-d H:i:s'),
                'icon' => '↩️',
                'completed' => true,
            ];
        }

        return $timeline;
    }
}