<?php
// app/Http\Controllers\Web\Employee\ReceiptController.php

namespace App\Http\Controllers\Web\Employee;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class ReceiptController extends BaseApiController
{
    /**
     * Display list of recent receipts
     */
    public function index()
    {
        $response = $this->apiRequest('get', 'employee/orders', [
            'status' => 'completed',
            'limit' => 20
        ]);
        
        if ($response['success']) {
            return view('employee.receipts.index', [
                'title' => 'Receipts',
                'orders' => $response['data']['data']['orders'] ?? $response['data']['orders'] ?? []
            ]);
        }
        
        return view('employee.receipts.index', [
            'title' => 'Receipts',
            'orders' => []
        ]);
    }

    /**
     * Show receipt for order
     */
    public function show($orderId)
    {
        $response = $this->apiRequest('get', "receipts/{$orderId}");
        
        if ($response['success']) {
            return view('employee.receipts.show', [
                'receipt' => $response['data']['data'] ?? $response['data'],
                'orderId' => $orderId,
            ]);
        }
        
        return redirect()->route('employee.orders.show', $orderId)
            ->with('error', 'Receipt not found');
    }

    /**
     * Print receipt
     */
    public function print($orderId)
    {
        $response = $this->apiRequest('get', "receipts/{$orderId}/print");
        
        if ($response['success']) {
            $receipt = $response['data']['data'] ?? $response['data'];
            
            return view('employee.receipts.print', [
                'receipt' => $receipt,
                'orderId' => $orderId,
            ]);
        }
        
        return redirect()->route('employee.orders.show', $orderId)
            ->with('error', 'Failed to generate receipt');
    }

    /**
     * Email receipt to customer
     */
    public function email(Request $request, $orderId)
    {
        $request->validate([
            'email' => 'required|email',
        ]);
        
        $response = $this->apiRequest('post', "receipts/{$orderId}/email", [
            'email' => $request->email,
        ]);
        
        if ($response['success']) {
            return redirect()->route('employee.receipts.show', $orderId)
                ->with('success', 'Receipt sent to ' . $request->email);
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to send receipt']
        )->withInput();
    }

    /**
     * Download receipt as PDF
     */
    public function download($orderId)
    {
        $response = $this->apiRequest('get', "receipts/{$orderId}/download");
        
        if ($response['success'] && isset($response['data']['pdf_url'])) {
            return redirect($response['data']['pdf_url']);
        }
        
        return redirect()->route('employee.receipts.show', $orderId)
            ->with('error', 'Failed to download receipt');
    }
}