<?php
// app/Http\Controllers\Web\Employee\PaymentConfirmationController.php

namespace App\Http\Controllers\Web\Employee;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class PaymentConfirmationController extends BaseApiController
{
    /**
     * Display all payments
     */
    public function index(Request $request)
    {
        $params = $request->only(['status', 'payment_method_id', 'date', 'search', 'page']);
        
        $response = $this->apiRequest('get', 'payments', $params);
        
        if ($response['success']) {
            return view('employee.payments.index', [
                'payments' => $response['data']['data'] ?? [],
                'pagination' => $response['data']['meta'] ?? null,
                'statuses' => ['pending', 'waiting_confirmation', 'confirmed', 'failed', 'refunded'],
                'filters' => $params,
            ]);
        }
        
        return view('employee.payments.index', [
            'payments' => [],
            'statuses' => [],
            'filters' => $params,
        ]);
    }

    /**
     * Show specific payment
     */
    public function show($id)
    {
        $response = $this->apiRequest('get', "payments/{$id}");
        
        if ($response['success']) {
            return view('employee.payments.show', [
                'payment' => $response['data']['data'] ?? $response['data'],
                'statuses' => ['pending', 'waiting_confirmation', 'confirmed', 'failed', 'refunded'],
            ]);
        }
        
        return redirect()->route('employee.payments.index')
            ->with('error', 'Payment not found');
    }

    /**
     * Confirm payment
     */
    public function confirm(Request $request, $id)
    {
        $request->validate([
            'confirmed_amount' => 'required|numeric|min:0',
            'reference_number' => 'nullable|string|max:100',
            'notes' => 'nullable|string|max:500',
        ]);
        
        $response = $this->apiRequest('put', "payments/{$id}/confirm", $request->all());
        
        if ($response['success']) {
            // Auto update order status if payment confirmed
            if (isset($response['data']['data']['order_id'])) {
                $this->apiRequest('put', "orders/{$response['data']['data']['order_id']}/status", [
                    'status' => 'confirmed',
                ]);
            }
            
            return redirect()->route('employee.payments.show', $id)
                ->with('success', 'Payment confirmed successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to confirm payment']
        )->withInput();
    }

    /**
     * Validate payment (check if proof is valid)
     */
    public function validatePayment(Request $request, $id)
    {
        $request->validate([
            'is_valid' => 'required|boolean',
            'validation_notes' => 'nullable|string|max:500',
        ]);
        
        $response = $this->apiRequest('put', "payments/{$id}/validate", $request->all());
        
        if ($response['success']) {
            $status = $request->is_valid ? 'validated' : 'rejected';
            return redirect()->route('employee.payments.show', $id)
                ->with('success', "Payment {$status} successfully");
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to validate payment']
        )->withInput();
    }

    /**
     * Reject payment
     */
    public function reject($id)
    {
        $response = $this->apiRequest('put', "payments/{$id}/validate", [
            'is_valid' => false,
            'validation_notes' => 'Payment proof rejected',
        ]);
        
        if ($response['success']) {
            return redirect()->route('employee.payments.show', $id)
                ->with('success', 'Payment rejected');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to reject payment']
        );
    }

    /**
     * Mark payment as failed
     */
    public function markAsFailed($id)
    {
        $response = $this->apiRequest('put', "payments/{$id}/confirm", [
            'status' => 'failed',
        ]);
        
        if ($response['success']) {
            return redirect()->route('employee.payments.show', $id)
                ->with('success', 'Payment marked as failed');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to update payment']
        );
    }
}