<?php
// app/Http\Controllers/Web/Employee/OrderManagementController.php

namespace App\Http\Controllers\Web\Employee;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class OrderManagementController extends BaseApiController
{
    /**
     * Display all orders
     */
    public function index(Request $request)
    {
        $params = $request->only(['status', 'date', 'search', 'page']);
        
        $response = $this->apiRequest('get', 'employee/orders', $params);
        
        return $this->handleResponse($response, 'employee.orders.index', [
            'title' => 'Order Management',
            'statuses' => ['pending', 'confirmed', 'preparing', 'ready', 'completed', 'cancelled'],
            'filters' => $params,
        ]);
    }

    /**
     * Display today's orders
     */
    public function todaysOrders()
    {
        $response = $this->apiRequest('get', 'employee/orders/today');
        
        return $this->handleResponse($response, 'employee.orders.today', [
            'title' => "Today's Orders"
        ]);
    }

    /**
     * Show specific order
     */
    public function show($id)
    {
        $response = $this->apiRequest('get', "employee/orders/{$id}");
        
        return $this->handleResponse($response, 'employee.orders.show', [
            'title' => 'Order Details',
            'statuses' => ['pending', 'confirmed', 'preparing', 'ready', 'completed', 'cancelled'],
        ]);
    }

    /**
     * Update order status
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:pending,confirmed,preparing,ready,completed,cancelled',
            'notes' => 'nullable|string|max:500',
        ]);
        
        $response = $this->apiRequest('put', "employee/orders/{$id}/status", [
            'status' => $request->status,
            'notes' => $request->notes,
        ]);
        
        return $this->handleRedirectResponse($response, 
            'employee.orders.show', 
            ['id' => $id],
            'Order status updated'
        );
    }

    /**
     * Mark order as ready
     */
    public function markAsReady($id)
    {
        $response = $this->apiRequest('put', "employee/orders/{$id}/status", [
            'status' => 'ready',
        ]);
        
        return $this->handleRedirectResponse($response, 
            'employee.orders.show', 
            ['id' => $id],
            'Order marked as ready'
        );
    }

    /**
     * Mark order as completed
     */
    public function markAsCompleted($id)
    {
        $response = $this->apiRequest('put', "employee/orders/{$id}/status", [
            'status' => 'completed',
        ]);
        
        return $this->handleRedirectResponse($response, 
            'employee.orders.show', 
            ['id' => $id],
            'Order marked as completed'
        );
    }
}