<?php
// app/Http\Controllers\Web\Customer\ReviewController.php

namespace App\Http\Controllers\Web\Customer;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class ReviewController extends BaseApiController
{
    /**
     * Display customer reviews
     */
    public function index()
    {
        $response = $this->apiRequest('get', 'reviews');
        
        if ($response['success']) {
            return view('customer.reviews.index', [
                'reviews' => $response['data']['data'] ?? [],
                'pagination' => $response['data']['meta'] ?? null,
            ]);
        }
        
        return view('customer.reviews.index', [
            'reviews' => [],
        ]);
    }

    /**
     * Show create review form
     */
    public function create($orderItemId = null)
    {
        // If order item ID provided, pre-fill the form
        $orderItem = null;
        if ($orderItemId) {
            $orderItemResponse = $this->apiRequest('get', "order-items/{$orderItemId}");
            if ($orderItemResponse['success']) {
                $orderItem = $orderItemResponse['data']['data'] ?? $orderItemResponse['data'];
            }
        }
        
        // Get orders that can be reviewed
        $ordersResponse = $this->apiRequest('get', 'orders?reviewable=true');
        
        return view('customer.reviews.create', [
            'orderItem' => $orderItem,
            'orders' => $ordersResponse['success'] ? 
                ($ordersResponse['data']['data'] ?? []) : [],
        ]);
    }

    /**
     * Store new review
     */
    public function store(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'menu_id' => 'required|exists:menus,id',
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:1000',
        ]);
        
        $response = $this->apiRequest('post', 'reviews', $request->all());
        
        if ($response['success']) {
            return redirect()->route('customer.reviews.index')
                ->with('success', 'Review submitted successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to submit review']
        )->withInput();
    }

    /**
     * Show specific review
     */
    public function show($id)
    {
        $response = $this->apiRequest('get', "reviews/{$id}");
        
        if ($response['success']) {
            return view('customer.reviews.show', [
                'review' => $response['data']['data'] ?? $response['data'],
            ]);
        }
        
        return redirect()->route('customer.reviews.index')
            ->with('error', 'Review not found');
    }

    /**
     * Show edit review form
     */
    public function edit($id)
    {
        $response = $this->apiRequest('get', "reviews/{$id}");
        
        if ($response['success']) {
            return view('customer.reviews.edit', [
                'review' => $response['data']['data'] ?? $response['data'],
            ]);
        }
        
        return redirect()->route('customer.reviews.index')
            ->with('error', 'Review not found');
    }

    /**
     * Update review
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:1000',
        ]);
        
        $response = $this->apiRequest('put', "reviews/{$id}", $request->all());
        
        if ($response['success']) {
            return redirect()->route('customer.reviews.show', $id)
                ->with('success', 'Review updated successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to update review']
        )->withInput();
    }

    /**
     * Delete review
     */
    public function destroy($id)
    {
        $response = $this->apiRequest('delete', "reviews/{$id}");
        
        if ($response['success']) {
            return redirect()->route('customer.reviews.index')
                ->with('success', 'Review deleted successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to delete review']
        );
    }
}