<?php
// app/Http\Controllers/Web\Customer\PaymentController.php

namespace App\Http\Controllers\Web\Customer;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class PaymentController extends BaseApiController
{
    /**
     * Show payment page for specific order
     */
    public function show($orderId)
    {
        $response = $this->apiRequest('get', "payments/{$orderId}");
        
        if ($response['success']) {
            return view('customer.payments.show', [
                'payment' => $response['data']['data'] ?? $response['data'],
                'orderId' => $orderId,
            ]);
        }
        
        return redirect()->route('customer.orders.index')
            ->with('error', 'Payment not found');
    }

    /**
     * Show create payment form
     */
    public function create($orderId)
    {
        // Get order details
        $orderResponse = $this->apiRequest('get', "orders/{$orderId}");
        // Get payment methods
        $methodsResponse = $this->apiRequest('get', 'payment-methods');
        
        if ($orderResponse['success']) {
            return view('customer.payments.create', [
                'order' => $orderResponse['data']['data'] ?? $orderResponse['data'],
                'paymentMethods' => $methodsResponse['success'] ? 
                    ($methodsResponse['data']['data'] ?? []) : [],
                'orderId' => $orderId,
            ]);
        }
        
        return redirect()->route('customer.orders.index')
            ->with('error', 'Order not found');
    }

    /**
     * Process payment
     */
    public function store(Request $request)
    {
        $request->validate([
            'order_id' => 'required|exists:orders,id',
            'payment_method_id' => 'required|exists:payment_methods,id',
            'amount' => 'required|numeric|min:0',
            'notes' => 'nullable|string|max:500',
        ]);
        
        $response = $this->apiRequest('post', 'payments', $request->all());
        
        if ($response['success']) {
            return redirect()->route('customer.orders.show', $request->order_id)
                ->with('success', 'Payment submitted successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to process payment']
        )->withInput();
    }

    /**
     * Upload payment proof
     */
    public function upload(Request $request)
    {
        $request->validate([
            'payment_id' => 'required|exists:payments,id',
            'proof_image' => 'required|image|mimes:jpg,jpeg,png|max:2048',
        ]);
        
        // Upload file via API
        if ($request->hasFile('proof_image')) {
            $uploadResponse = $this->apiRequest('post', 'upload', [
                'file' => $request->file('proof_image'),
            ]);
            
            if ($uploadResponse['success']) {
                // Update payment with proof image
                $updateResponse = $this->apiRequest('put', "payments/{$request->payment_id}", [
                    'proof_image' => $uploadResponse['data']['path'] ?? $uploadResponse['data']['url'],
                    'status' => 'waiting_confirmation',
                ]);
                
                if ($updateResponse['success']) {
                    return redirect()->route('customer.orders.index')
                        ->with('success', 'Payment proof uploaded successfully');
                }
            }
        }
        
        return back()->withErrors([
            'error' => 'Failed to upload payment proof',
        ])->withInput();
    }

    /**
     * Show upload proof form
     */
    public function showUploadForm($paymentId)
    {
        $response = $this->apiRequest('get', "payments/{$paymentId}");
        
        if ($response['success']) {
            return view('customer.payments.upload', [
                'payment' => $response['data']['data'] ?? $response['data'],
            ]);
        }
        
        return redirect()->route('customer.orders.index')
            ->with('error', 'Payment not found');
    }
}