<?php

namespace App\Http\Controllers\Web\Customer;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Cart;
use App\Models\Payment;
use App\Models\Review;

class OrderController extends Controller
{
    /**
     * Display customer orders
     */
    public function index(Request $request)
    {
        // Check authentication
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Please login to view your orders');
        }

        $perPage = $request->get('limit', 10);
        $status = $request->get('status');

        // Get orders with relationships
        $query = Order::where('customer_id', Auth::id())
            ->with(['items.menu', 'payments'])
            ->orderBy('created_at', 'desc');

        // Filter by status if provided
        if ($status) {
            $query->where('status', $status);
        }

        $orders = $query->paginate($perPage);

        // Calculate stats
        $stats = [
            'total' => Order::where('customer_id', Auth::id())->count(),
            'pending' => Order::where('customer_id', Auth::id())->where('status', 'pending')->count(),
            'completed' => Order::where('customer_id', Auth::id())->where('status', 'completed')->count(),
            'cancelled' => Order::where('customer_id', Auth::id())->where('status', 'cancelled')->count(),
        ];

        return view('customer.orders.index', [
            'title' => 'My Orders',
            'orders' => $orders,
            'stats' => $stats,
            'statusColors' => $this->getStatusColors(),
        ]);
    }

    /**
     * Show specific order
     */
    public function show($id)
    {
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Please login to view order details');
        }

        $order = Order::where('customer_id', Auth::id())
            ->with(['items.menu', 'payments', 'customer'])
            ->findOrFail($id);

        return view('customer.orders.show', [
            'title' => 'Order #' . $order->order_number,
            'order' => $order,
            'statusColors' => $this->getStatusColors(),
        ]);
    }

    /**
     * Show create order form (checkout page)
     */
    public function create()
    {
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Please login to place an order');
        }

        // Get cart items
        $cartItems = Cart::where('user_id', Auth::id())
            ->with('menu')
            ->get();

        if ($cartItems->isEmpty()) {
            return redirect()->route('customer.cart.index')
                ->with('error', 'Your cart is empty');
        }

        // Get active payment methods
        $paymentMethods = \App\Models\PaymentMethod::where('is_active', 1)->get();

        // Calculate totals
        $subtotal = $cartItems->sum(function ($item) {
            return $item->menu->price * $item->quantity;
        });
        $taxAmount = $subtotal * 0.10;
        $serviceFee = 2000;
        $total = $subtotal + $taxAmount + $serviceFee;

        return view('customer.orders.create', [
            'title' => 'Checkout',
            'cartItems' => $cartItems,
            'paymentMethods' => $paymentMethods,
            'subtotal' => $subtotal,
            'taxAmount' => $taxAmount,
            'serviceFee' => $serviceFee,
            'total' => $total,
        ]);
    }

    /**
     * Create new order (from cart)
     */
    public function store(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Please login to place an order');
        }

        // Build validation rules
        $rules = [
            'order_type' => 'required|in:dine_in,takeaway,delivery',
            'payment_method_id' => 'required|integer|exists:payment_methods,id',
            'notes' => 'nullable|string|max:500',
        ];

        // Delivery address only required for delivery orders
        if ($request->order_type === 'delivery') {
            $rules['delivery_address'] = 'required|string|min:10';
        } else {
            $rules['delivery_address'] = 'nullable|string';
        }

        // Validate all fields at once
        $validated = $request->validate($rules);

        DB::beginTransaction();
        try {
            // Get cart items
            $cartItems = Cart::where('user_id', Auth::id())
                ->with('menu')
                ->get();

            if ($cartItems->isEmpty()) {
                return back()->with('error', 'Your cart is empty')
                    ->withInput();
            }

            // Calculate totals
            $subtotal = $cartItems->sum(function ($item) {
                return $item->menu->price * $item->quantity;
            });
            $taxAmount = $subtotal * 0.10; // 10% tax
            $serviceFee = 2000; // Fixed service fee
            $finalAmount = $subtotal + $taxAmount + $serviceFee;

            // Generate order number
            $orderNumber = 'ORD-' . date('Ymd') . '-' . str_pad(Order::whereDate('created_at', today())->count() + 1, 4, '0', STR_PAD_LEFT);

            // Create order
            $order = Order::create([
                'order_number' => $orderNumber,
                'customer_id' => Auth::id(),
                'order_type' => $validated['order_type'],
                'status' => 'pending',
                'total_amount' => $subtotal,
                'tax_amount' => $taxAmount,
                'service_fee' => $serviceFee,
                'discount_amount' => 0,
                'final_amount' => $finalAmount,
                'notes' => $validated['notes'] ?? null,
            ]);

            // Create order items
            foreach ($cartItems as $cartItem) {
                OrderItem::create([
                    'order_id' => $order->id,
                    'menu_id' => $cartItem->menu_id,
                    'quantity' => $cartItem->quantity,
                    'unit_price' => $cartItem->menu->price,
                    'subtotal' => $cartItem->menu->price * $cartItem->quantity,
                    'notes' => $cartItem->notes,
                ]);
            }

            // Create payment record
            \App\Models\Payment::create([
                'order_id' => $order->id,
                'payment_method_id' => $validated['payment_method_id'],
                'amount' => $finalAmount,
                'status' => 'pending',
                'payment_date' => now(),
            ]);

            // Clear cart
            Cart::where('user_id', Auth::id())->delete();

            DB::commit();

            return redirect()->route('customer.orders.show', $order->id)
                ->with('success', 'Order placed successfully! Order #' . $orderNumber);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Failed to create order', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'user_id' => Auth::id()
            ]);

            return back()->with('error', 'Failed to create order: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Cancel an order
     */
    public function cancel(Request $request, $id)
    {
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Please login to cancel order');
        }

        $order = Order::where('customer_id', Auth::id())->findOrFail($id);

        // Only allow cancellation for pending orders
        if (!in_array($order->status, ['pending', 'confirmed'])) {
            return redirect()->route('customer.orders.show', $id)
                ->with('error', 'This order cannot be cancelled');
        }

        $order->update(['status' => 'cancelled']);

        return redirect()->route('customer.orders.show', $id)
            ->with('success', 'Order has been cancelled successfully');
    }

    /**
     * Get status color mapping
     */
    private function getStatusColors()
    {
        return [
            'pending' => 'warning',
            'confirmed' => 'info',
            'preparing' => 'primary',
            'ready' => 'success',
            'on_delivery' => 'info',
            'completed' => 'success',
            'cancelled' => 'danger',
            'delivered' => 'success'
        ];
    }

    // Tambahkan method-method berikut di OrderController:

    /**
     * Show rate order form
     */
    public function rate($id)
    {
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Please login to rate order');
        }

        $order = Order::where('customer_id', Auth::id())
            ->with(['items.menu'])
            ->findOrFail($id);

        // Check if order is completed
        if ($order->status !== 'completed') {
            return redirect()->route('customer.orders.show', $id)
                ->with('error', 'Only completed orders can be rated');
        }

        // Check if already rated
        $existingReview = Review::where('order_id', $order->id)
            ->where('customer_id', Auth::id())
            ->first();

        if ($existingReview) {
            return redirect()->route('customer.orders.show', $id)
                ->with('info', 'You have already rated this order');
        }

        return view('customer.orders.rate', [
            'title' => 'Rate Order #' . $order->order_number,
            'order' => $order,
        ]);
    }

    /**
     * Submit rating
     */
    public function submitRating(Request $request, $id)
    {
        if (!Auth::check()) {
            return redirect()->route('login')
                ->with('error', 'Please login to rate order');
        }

        $order = Order::where('customer_id', Auth::id())
            ->with(['items.menu'])
            ->findOrFail($id);

        // Check if order is completed
        if ($order->status !== 'completed') {
            return redirect()->route('customer.orders.show', $id)
                ->with('error', 'Only completed orders can be rated');
        }

        // Check if already rated
        $existingReview = Review::where('order_id', $order->id)
            ->where('customer_id', Auth::id())
            ->first();

        if ($existingReview) {
            return redirect()->route('customer.orders.show', $id)
                ->with('info', 'You have already rated this order');
        }

        // Validate
        $request->validate([
            'rating' => 'required|integer|min:1|max:5',
            'comment' => 'nullable|string|max:1000',
            'delivery_rating' => 'nullable|integer|min:1|max:5',
            'delivery_review' => 'nullable|string|max:500',
        ]);

        DB::beginTransaction();
        try {
            // Update order with rating
            if ($request->has('delivery_rating')) {
                $order->update([
                    'delivery_rating' => $request->delivery_rating,
                    'delivery_comment' => $request->delivery_comment,
                ]);
            }

            // Create review for each item
            if ($request->has('items')) {
                foreach ($request->items as $itemData) {
                    if (isset($itemData['item_id']) && isset($itemData['rating'])) {
                        $orderItem = $order->items()->find($itemData['item_id']);
                        if ($orderItem) {
                            Review::create([
                                'order_id' => $order->id,
                                'customer_id' => Auth::id(),
                                'menu_id' => $orderItem->menu_id,
                                'rating' => $itemData['rating'],
                                'comment' => $itemData['comment'] ?? null,
                                'is_approved' => true, // Auto approve for now
                            ]);

                            // Update menu average rating
                            $this->updateMenuRating($orderItem->menu_id);
                        }
                    }
                }
            }

            DB::commit();

            return redirect()->route('customer.orders.show', $id)
                ->with('success', 'Thank you for your rating!');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Failed to submit rating', [
                'error' => $e->getMessage(),
                'order_id' => $id,
                'user_id' => Auth::id()
            ]);

            return back()
                ->with('error', 'Failed to submit rating: ' . $e->getMessage())
                ->withInput();
        }
    }

/**
 * Show rating history
 */
public function ratingHistory(Request $request)
{
    \Log::info('=== RATING HISTORY ACCESSED ===');
    
    if (!Auth::check()) {
        \Log::warning('User not authenticated');
        return redirect()->route('login')
            ->with('error', 'Please login to view rating history');
    }

    \Log::info('User authenticated', [
        'user_id' => Auth::id(),
        'user_name' => Auth::user()->name,
        'user_role' => Auth::user()->role
    ]);

    try {
        $perPage = $request->get('limit', 10);
        
        // Get reviews with relationships
        $reviews = Review::where('customer_id', Auth::id())
            ->with(['order', 'menu'])
            ->orderBy('created_at', 'desc')
            ->paginate($perPage);
        
        \Log::info('Reviews loaded', ['count' => $reviews->count()]);
        
        // Calculate stats
        $stats = [
            'total' => Review::where('customer_id', Auth::id())->count(),
            'average' => round(Review::where('customer_id', Auth::id())->avg('rating') ?? 0, 1),
            'recent' => Review::where('customer_id', Auth::id())
                ->where('created_at', '>=', now()->subDays(30))
                ->count(),
        ];
        
        \Log::info('Stats calculated', $stats);
        
        return view('customer.orders.rating-history', [
            'title' => 'My Rating History',
            'reviews' => $reviews,
            'stats' => $stats,
        ]);
            
    } catch (\Exception $e) {
        \Log::error('Error in ratingHistory: ' . $e->getMessage(), [
            'trace' => $e->getTraceAsString()
        ]);
        
        // Fallback view
        return view('customer.orders.rating-history', [
            'title' => 'My Rating History',
            'reviews' => collect([]),
            'stats' => ['total' => 0, 'average' => 0, 'recent' => 0],
        ]);
    }
}

    /**
     * Helper to update menu average rating
     */
    private function updateMenuRating($menuId)
    {
        $reviews = Review::where('menu_id', $menuId)
            ->where('is_approved', true);

        $avgRating = $reviews->avg('rating');
        $ratingCount = $reviews->count();

        \App\Models\Menu::where('id', $menuId)->update([
            'average_rating' => $avgRating ? round($avgRating, 1) : 0,
            'rating_count' => $ratingCount
        ]);
    }
}
