<?php
// app/Http\Controllers/Web/Customer/MenuController.php

namespace App\Http\Controllers\Web\Customer;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Menu;
use App\Models\Category;

class MenuController extends Controller
{
    /**
     * Display customer menu page - PUBLIC (no auth needed)
     */
    public function index(Request $request)
    {
        // Ambil kategori UNIK yang memiliki menu aktif
        $categories = Category::where('is_active', true)
            ->whereHas('menus', function($query) {
                $query->where('is_available', true);
            })
            ->distinct('name') // Tambahkan distinct
            ->orderBy('name')
            ->get()
            ->unique('name'); // Tambahkan unique untuk memastikan

        // Ambil menu aktif tanpa duplikat
        $menus = Menu::where('is_available', true)
            ->distinct() // Tambahkan distinct
            ->orderBy('name')
            ->get()
            ->unique(function($item) {
                return $item->name . $item->price . $item->category_id;
            }); // Hindari duplikat berdasarkan nama, harga, dan kategori

        // Jika masih ada duplikat, group by name
        $menus = $menus->groupBy('name')->map(function($group) {
            return $group->first(); // Ambil hanya satu dari setiap nama
        })->values();

        return view('customer.menu.index', [
            'title' => 'Our Menu',
            'categories' => $categories,
            'menus' => $menus,
            'pagination' => null,
            'filters' => []
        ]);
    }

    /**
     * Display menu details - PUBLIC
     */
    public function show($id)
    {
        $menu = Menu::with(['category', 'reviews' => function($query) {
            $query->where('is_approved', true)->latest();
        }])->findOrFail($id);

        return view('customer.menu.show', [
            'title' => 'Menu Details',
            'menu' => $menu
        ]);
    }

    /**
     * Get special menus - PUBLIC
     */
    public function special()
    {
        $menus = Menu::where('is_special', true)
            ->where('is_available', true)
            ->with('category')
            ->get();

        return view('customer.menu.special', [
            'title' => 'Special Menus',
            'menus' => $menus
        ]);
    }

    /**
     * Get best sellers - PUBLIC
     */
    public function bestSeller()
    {
        $menus = Menu::where('is_best_seller', true)
            ->where('is_available', true)
            ->with('category')
            ->get();

        return view('customer.menu.best-seller', [
            'title' => 'Best Seller Menus',
            'menus' => $menus
        ]);
    }

    /**
     * Get menu categories - AJAX
     */
    public function categories()
    {
        $categories = Category::where('is_active', true)
            ->whereHas('menus', function($query) {
                $query->where('is_available', true);
            })
            ->orderBy('name')
            ->get()
            ->unique('name')
            ->values();
            
        return response()->json($categories);
    }

    /**
     * Search menus - AJAX
     */
    public function search(Request $request)
    {
        $query = Menu::where('is_available', true)->with('category');
        
        if ($search = $request->input('query')) {
            $query->where(function($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }
        
        if ($categoryId = $request->input('category')) {
            $query->where('category_id', $categoryId);
        }
        
        $menus = $query->get()->unique('id')->values();

        return response()->json([
            'success' => true,
            'data' => $menus
        ]);
    }
}
