<?php
// app/Http\Controllers/Web/Customer/CartController.php

namespace App\Http\Controllers\Web\Customer;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Cart;
use App\Models\Menu;
use Illuminate\Support\Facades\Auth;

class CartController extends Controller
{
    public function index()
    {
        $cartItems = Cart::where('user_id', Auth::id())
            ->with('menu.category')
            ->get();
        
        $subtotal = $cartItems->sum(function($item) {
            return $item->quantity * $item->menu->price;
        });
        
        $tax = $subtotal * 0.1;
        $serviceFee = 2000;
        $total = $subtotal + $tax + $serviceFee;
        
        // Format data untuk view
        $items = $cartItems->map(function($item) {
            return [
                'id' => $item->menu_id, // Use menu_id for route params
                'menu_id' => $item->menu_id,
                'quantity' => $item->quantity,
                'notes' => $item->notes,
                'unit_price' => $item->menu->price,
                'subtotal' => $item->quantity * $item->menu->price,
                'menu' => [
                    'id' => $item->menu->id,
                    'name' => $item->menu->name,
                    'price' => $item->menu->price,
                    'image_url' => $item->menu->image_url,
                    'is_available' => $item->menu->is_available,
                ]
            ];
        });
        
        $cart = [
            'subtotal' => $subtotal,
            'tax_amount' => $tax,
            'service_fee' => $serviceFee,
            'total' => $total,
            'item_count' => $cartItems->count()
        ];
        
        return view('customer.cart.index', [
            'title' => 'My Cart',
            'cartItems' => $cartItems,
            'items' => $items,
            'cart' => $cart,
            'total' => $total
        ]);
    }
    
    public function store(Request $request)
    {
        $request->validate([
            'menu_id' => 'required|integer|exists:menus,id',
            'quantity' => 'required|integer|min:1',
            'notes' => 'nullable|string|max:500',
        ]);
        
        $menu = Menu::findOrFail($request->menu_id);
        
        if (!$menu->is_available) {
            // Return JSON for AJAX requests
            if ($request->expectsJson() || $request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Menu tidak tersedia'
                ], 422);
            }
            return redirect()->back()->with('error', 'Menu tidak tersedia');
        }
        
        // Check if item already exists in cart
        $existingCart = Cart::where('user_id', Auth::id())
            ->where('menu_id', $request->menu_id)
            ->first();
        
        if ($existingCart) {
            // Increment quantity instead of replacing
            $existingCart->quantity += $request->quantity;
            $existingCart->notes = $request->notes ?? $existingCart->notes;
            $existingCart->save();
            $cartItem = $existingCart;
        } else {
            // Create new cart item
            $cartItem = Cart::create([
                'user_id' => Auth::id(),
                'menu_id' => $request->menu_id,
                'quantity' => $request->quantity,
                'notes' => $request->notes
            ]);
        }
        
        // Return JSON for AJAX requests
        if ($request->expectsJson() || $request->ajax()) {
            // Get distinct cart items count (number of different products)
            $cartCount = Cart::where('user_id', Auth::id())->count();
            
            return response()->json([
                'success' => true,
                'message' => 'Item berhasil ditambahkan ke keranjang',
                'data' => [
                    'cart_count' => $cartCount,
                    'item' => $cartItem
                ]
            ]);
        }
        
        // Get menu name for success message
        $menuName = $menu->name;
        
        // Redirect back to menu page with success message
        return redirect()->back()
            ->with('success', "✓ {$menuName} telah ditambahkan ke keranjang!");
    }
    
    public function update(Request $request, $menu_id)
    {
        $request->validate([
            'quantity' => 'required|integer|min:1',
            'notes' => 'nullable|string|max:500',
        ]);
        
        $cart = Cart::where('user_id', Auth::id())
            ->where('menu_id', $menu_id)
            ->firstOrFail();
        
        $cart->update([
            'quantity' => $request->quantity,
            'notes' => $request->notes
        ]);
        
        // Return JSON for AJAX requests
        if ($request->expectsJson() || $request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Cart updated successfully'
            ]);
        }
        
        return redirect()->route('customer.cart.index')
            ->with('success', 'Cart updated');
    }
    
    public function destroy($menu_id)
    {
        Cart::where('user_id', Auth::id())
            ->where('menu_id', $menu_id)
            ->delete();
        
        return redirect()->route('customer.cart.index')
            ->with('success', 'Item removed from cart');
    }
    
    public function clear()
    {
        Cart::where('user_id', Auth::id())->delete();
        
        return redirect()->route('customer.cart.index')
            ->with('success', 'Cart cleared');
    }
    
    public function checkoutSummary()
    {
        $cartItems = Cart::where('user_id', Auth::id())
            ->with('menu')
            ->get();
        
        if ($cartItems->isEmpty()) {
            return redirect()->route('customer.cart.index')
                ->with('error', 'Cart is empty');
        }
        
        $subtotal = $cartItems->sum(function($item) {
            return $item->quantity * $item->menu->price;
        });
        
        $tax = $subtotal * 0.1; // 10% tax
        $serviceFee = 2000; // Flat service fee
        $total = $subtotal + $tax + $serviceFee;
        
        if (request()->ajax()) {
            return response()->json([
                'success' => true,
                'data' => [
                    'items' => $cartItems,
                    'subtotal' => $subtotal,
                    'tax' => $tax,
                    'service_fee' => $serviceFee,
                    'total' => $total
                ]
            ]);
        }
        
        return view('customer.cart.checkout', [
            'title' => 'Checkout Summary',
            'cartItems' => $cartItems,
            'subtotal' => $subtotal,
            'tax' => $tax,
            'serviceFee' => $serviceFee,
            'total' => $total
        ]);
    }
}
