<?php
// app/Http\Controllers\Web\Customer\CareerController.php

namespace App\Http\Controllers\Web\Customer;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class CareerController extends BaseApiController
{
    /**
     * Display public careers page
     */
    public function publicIndex()
    {
        $response = $this->apiRequest('get', 'careers');
        
        return view('customer.careers.index', [
            'careers' => $response['success'] ? 
                ($response['data']['data'] ?? []) : [],
            'isPublic' => true,
        ]);
    }

    /**
     * Display customer's career applications
     */
    public function index()
    {
        $response = $this->apiRequest('get', 'careers');
        
        if ($response['success']) {
            return view('customer.careers.index', [
                'careers' => $response['data']['data'] ?? [],
                'isPublic' => false,
            ]);
        }
        
        return view('customer.careers.index', [
            'careers' => [],
            'isPublic' => false,
        ]);
    }

    /**
     * Show specific career details
     */
    public function show($id)
    {
        $response = $this->apiRequest('get', "careers/{$id}");
        
        if ($response['success']) {
            return view('customer.careers.show', [
                'career' => $response['data']['data'] ?? $response['data'],
            ]);
        }
        
        return redirect()->route('customer.careers.index')
            ->with('error', 'Career not found');
    }

    /**
     * Show application form
     */
    public function apply($id)
    {
        $response = $this->apiRequest('get', "careers/{$id}");
        
        if ($response['success']) {
            return view('customer.careers.apply', [
                'career' => $response['data']['data'] ?? $response['data'],
            ]);
        }
        
        return redirect()->route('customer.careers.index')
            ->with('error', 'Career not found');
    }

    /**
     * Submit application
     */
    public function submitApplication(Request $request, $id)
    {
        $request->validate([
            'applicant_name' => 'required|string|max:255',
            'applicant_email' => 'required|email',
            'applicant_phone' => 'required|string|max:20',
            'cover_letter' => 'nullable|string|max:2000',
            'cv' => 'required|file|mimes:pdf,doc,docx|max:5120', // 5MB max
        ]);
        
        // Prepare data
        $data = $request->except('cv');
        
        // Upload CV via API
        if ($request->hasFile('cv')) {
            $uploadResponse = $this->apiRequest('post', 'upload', [
                'file' => $request->file('cv'),
            ]);
            
            if ($uploadResponse['success']) {
                $data['applicant_cv'] = $uploadResponse['data']['path'] ?? $uploadResponse['data']['url'];
            } else {
                return back()->withErrors([
                    'cv' => 'Failed to upload CV',
                ])->withInput();
            }
        }
        
        $response = $this->apiRequest('post', "careers/{$id}/apply", $data);
        
        if ($response['success']) {
            return redirect()->route('customer.careers.applications.index')
                ->with('success', 'Application submitted successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to submit application']
        )->withInput();
    }

    /**
     * Display my applications
     */
    public function myApplications()
    {
        $response = $this->apiRequest('get', 'careers/applications');
        
        if ($response['success']) {
            return view('customer.careers.applications', [
                'applications' => $response['data']['data'] ?? [],
                'pagination' => $response['data']['meta'] ?? null,
            ]);
        }
        
        return view('customer.careers.applications', [
            'applications' => [],
        ]);
    }

    /**
     * Show specific application
     */
    public function showApplication($id)
    {
        $response = $this->apiRequest('get', "careers/applications/{$id}");
        
        if ($response['success']) {
            return view('customer.careers.application-show', [
                'application' => $response['data']['data'] ?? $response['data'],
            ]);
        }
        
        return redirect()->route('customer.careers.applications.index')
            ->with('error', 'Application not found');
    }

    /**
     * Cancel application
     */
    public function cancelApplication($id)
    {
        // Note: API might not have cancel endpoint, you might need to update status
        $response = $this->apiRequest('put', "careers/applications/{$id}", [
            'status' => 'withdrawn',
        ]);
        
        if ($response['success']) {
            return redirect()->route('customer.careers.applications.index')
                ->with('success', 'Application cancelled successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to cancel application']
        );
    }
}