<?php
// app/Http\Controllers/Web/Auth/EmployeeAuthController.php

namespace App\Http\Controllers\Web\Auth;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class EmployeeAuthController extends BaseApiController
{
    /**
     * Show employee login form
     */
    public function showLoginForm()
    {
        return view('auth.employee-login');
    }

    /**
     * Process employee login via API
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // PERBAIKAN: Gunakan endpoint yang benar dari routes/api.php
        $response = $this->apiRequest('post', 'auth/login', [
            'email' => $request->email,
            'password' => $request->password,
        ]);

        if ($response['success'] && $response['data']['success']) {
            $data = $response['data']['data'];
            
            // Cek apakah user adalah employee
            if ($data['user']['role'] !== 'employee') {
                return back()->withErrors([
                    'email' => 'Access denied. Employees only.',
                ])->withInput();
            }
            
            // Simpan data di session
            Session::put('api_token', $data['token']);
            Session::put('user', $data['user']);
            Session::put('user_role', $data['user']['role']);
            Session::put('is_employee', true);

            return redirect()->route('employee.dashboard')
                ->with('success', 'Welcome, ' . $data['user']['name']);
        }

        $errorMessage = $response['data']['message'] ?? 'Invalid credentials';
        return back()->withErrors([
            'email' => $errorMessage,
        ])->withInput($request->only('email'));
    }

    /**
     * Show employee profile
     */
    public function profile()
    {
        $response = $this->apiRequest('get', 'auth/me');

        if ($response['success'] && $response['data']['success']) {
            return view('employee.profile', [
                'user' => $response['data']['data'],
            ]);
        }

        return redirect()->route('employee.dashboard')
            ->with('error', 'Failed to load profile');
    }

    /**
     * Update employee profile
     */
    public function updateProfile(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'phone' => 'required|string|max:20',
            'current_password' => 'nullable|string|min:6',
            'new_password' => 'nullable|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $data = $request->only(['name', 'email', 'phone']);

        if ($request->filled('current_password') && $request->filled('new_password')) {
            $data['current_password'] = $request->current_password;
            $data['new_password'] = $request->new_password;
        }

        $response = $this->apiRequest('put', 'auth/profile', $data);

        if ($response['success'] && $response['data']['success']) {
            // Update session data
            if (isset($response['data']['data'])) {
                Session::put('user', $response['data']['data']);
            }

            return redirect()->route('employee.profile')
                ->with('success', 'Profile updated successfully');
        }

        $errorMessage = $response['data']['message'] ?? 'Failed to update profile';
        return back()->with('error', $errorMessage)->withInput();
    }

    /**
     * Employee logout
     */
    public function logout()
    {
        // Panggil API logout
        $this->apiRequest('post', 'auth/logout');

        // Clear session
        Session::flush();

        return redirect()->route('employee.login.form')
            ->with('success', 'Logged out successfully');
    }

    /**
     * Show employee dashboard (tambahan)
     */
    public function dashboard()
    {
        $response = $this->apiRequest('get', 'employee/auth/dashboard');

        if ($response['success'] && $response['data']['success']) {
            return view('employee.dashboard', [
                'dashboardData' => $response['data']['data'],
            ]);
        }

        return view('employee.dashboard', [
            'dashboardData' => [],
            'error' => 'Failed to load dashboard data'
        ]);
    }
}