<?php

namespace App\Http\Controllers\Web\Auth;

use App\Http\Controllers\Controller;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    /**
     * Show login form
     */
    public function showLoginForm()
    {
        // Jika user sudah login, redirect ke dashboard
        if (Auth::check()) {
            return $this->redirectBasedOnRole();
        }
        
        // Clear session flags untuk login page
        Session::forget('login_attempts');
        
        return view('auth.login');
    }

    /**
     * Handle login
     */
    public function login(Request $request)
    {
        // Jika sudah login, langsung redirect
        if (Auth::check()) {
             \Log::info('User already logged in, redirecting based on role');
            return $this->redirectBasedOnRole();
        }

        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Cari user
        $user = User::where('email', $request->email)->first();

        if (!$user) {
            return back()->with('error', 'Email tidak ditemukan')->withInput();
        }

        // Cek password
        if (!Hash::check($request->password, $user->password)) {
            return back()->with('error', 'Password salah')->withInput();
        }

            // Debug: Log role user sebelum login
    \Log::info('User login attempt', [
        'user_id' => $user->id,
        'email' => $user->email,
        'role' => $user->role,
        'expected_redirect' => $user->role == 'customer' ? 'customer.dashboard' : 'admin.dashboard'
    ]);

        // Login dengan Laravel Auth
        Auth::login($user, $request->has('remember'));

        // ✅ SIMPAN SESSION DATA UNTUK NAVBAR
        Session::put('api_token', 'web_auth_' . $user->id); // Untuk kompatibilitas dengan kode lama
        Session::put('user', $user->toArray()); // Simpan sebagai array
        Session::put('user_role', $user->role);
        Session::put('user_id', $user->id);
        Session::put('is_logged_in', true);

        // Regenerate session untuk security
        $request->session()->regenerate();

            // Debug setelah login
    \Log::info('User logged in successfully', [
        'user_id' => $user->id,
        'role' => $user->role,
        'session_role' => session('user_role'),
        'auth_role' => Auth::user()->role
    ]);

        // Redirect berdasarkan role
        return $this->redirectBasedOnRole($user);
    }

    /**
     * Helper untuk redirect berdasarkan role
     */
    private function redirectBasedOnRole($user = null)
    {
        $user = $user ?? Auth::user();
        
        if (!$user) {
            return redirect()->route('home');
        }

            // Debug role
    \Log::info('Redirecting user based on role', [
        'user_id' => $user->id,
        'role' => $user->role,
        'name' => $user->name
    ]);

        $redirect = match ($user->role) {
            'admin' => redirect()->route('admin.dashboard')->with('success', 'Welcome back, ' . $user->name),
            'employee' => redirect()->route('employee.dashboard')->with('success', 'Welcome back, ' . $user->name),
            'customer' => redirect()->route('customer.dashboard')->with('success', 'Welcome back, ' . $user->name),
            default => redirect()->route('home')->with('success', 'Login successful'),
        };

        return $redirect;
    }

    /**
     * Menampilkan halaman profil
     */
    public function profile()
    {
        // Jika tidak login, redirect ke login
        if (!Auth::check()) {
            return redirect()->route('login')->with('error', 'Please login first');
        }

        // Ambil user dari auth (lebih reliable)
        $user = Auth::user();

        return view('customer.profile.index', [
            'user' => $user
        ]);
    }

        /**
     * Show edit profile form
     */
    public function editProfile()
    {
        if (!Auth::check()) {
            return redirect()->route('login')->with('error', 'Please login first');
        }

        $user = Auth::user();

        return view('customer.profile.edit', [
            'user' => $user
        ]);
    }

    /**
     * Update profile
     */
    public function updateProfile(Request $request)
    {
        if (!Auth::check()) {
            return redirect()->route('login');
        }

        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'required|string|max:20',
            'current_password' => 'nullable|string|min:6',
            'new_password' => 'nullable|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return redirect()->route('customer.profile.edit')
                ->withErrors($validator)
                ->withInput();
        }

        try {
            // Update basic info
            $user->name = $request->name;
            $user->email = $request->email;
            $user->phone = $request->phone;

            // Update password jika diminta
            if ($request->filled('current_password') && $request->filled('new_password')) {
                if (!Hash::check($request->current_password, $user->password)) {
                    return redirect()->route('customer.profile.edit')
                        ->with('error', 'Current password is incorrect')
                        ->withInput();
                }
                $user->password = Hash::make($request->new_password);
            }

            $user->save();

            // Update session user data
            Session::put('user', $user->toArray());

            // Redirect ke profile index setelah berhasil update
            return redirect()->route('customer.profile.index')
                ->with('success', 'Profile updated successfully');

        } catch (\Exception $e) {
            \Log::error('Failed to update profile: ' . $e->getMessage());
            
            return redirect()->route('customer.profile.edit')
                ->with('error', 'Failed to update profile: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Show registration form
     */
    public function showRegistrationForm()
    {
        // Jika sudah login, redirect ke dashboard
        if (Auth::check()) {
            return $this->redirectBasedOnRole();
        }

        return view('auth.register');
    }

    /**
     * Handle registration
     */
    public function register(Request $request)
    {
        // Jika sudah login, redirect ke dashboard
        if (Auth::check()) {
            return $this->redirectBasedOnRole();
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|max:20',
            'password' => 'required|string|min:6|confirmed',
            'terms' => 'accepted',
        ], [
            'terms.accepted' => 'You must accept the terms and conditions'
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Create user
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            'role' => 'customer',
        ]);

        // Auto login setelah registrasi
        Auth::login($user);
        
        // Simpan session data untuk navbar
        Session::put('api_token', 'web_auth_' . $user->id);
        Session::put('user', $user->toArray());
        Session::put('user_role', $user->role);
        Session::put('user_id', $user->id);
        Session::put('is_logged_in', true);

        // Regenerate session
        $request->session()->regenerate();

        return redirect()->route('customer.dashboard')
            ->with('success', 'Registration successful! Welcome, ' . $user->name);
    }

    /**
     * Handle logout
     */
    public function logout(Request $request)
    {
        if (Auth::check()) {
            Auth::logout();
        }

        // Clear semua session data
        Session::flush();
        
        // Invalidate session
        $request->session()->invalidate();
        
        // Regenerate CSRF token
        $request->session()->regenerateToken();

        return redirect()->route('home')
            ->with('success', 'Logged out successfully');
    }


    /**
     * Clear session and show login (for debugging)
     */
    public function clearAndLogin(Request $request)
    {
        // Clear semua
        Auth::logout();
        Session::flush();
        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login')
            ->with('info', 'Session cleared. Please login.');
    }
}