<?php
// app/Http/Controllers/Web/Auth/AdminAuthController.php

namespace App\Http\Controllers\Web\Auth;

use App\Http\Controllers\Web\Api\BaseApiController;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;

class AdminAuthController extends BaseApiController
{
    /**
     * Show admin login form
     */
    public function showLoginForm()
    {
        return view('auth.admin-login');
    }

    /**
     * Process admin login
     */
    public function login(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Cari user berdasarkan email
        $user = User::where('email', $request->email)->first();
        
        if (!$user) {
            return back()->withErrors([
                'email' => 'Email tidak ditemukan',
            ])->withInput();
        }
        
        // Cek password
        if (!Hash::check($request->password, $user->password)) {
            return back()->withErrors([
                'password' => 'Password salah',
            ])->withInput();
        }
        
        // Cek apakah user adalah admin
        if ($user->role !== 'admin') {
            return back()->withErrors([
                'email' => 'Access denied. Admin only.',
            ])->withInput();
        }
        
        // Login dengan Laravel Auth
        Auth::login($user);
        
        // ✅ SIMPAN SESSION UNTUK MIDDLEWARE
        Session::put('api_token', 'web_auth_admin_' . $user->id); // Token untuk middleware
        Session::put('user', $user);
        Session::put('user_role', $user->role);
        Session::put('is_admin', true);
        Session::put('is_logged_in', true);

        // Redirect ke admin dashboard
        return redirect()->route('admin.dashboard')
            ->with('success', 'Welcome back, ' . $user->name);
    }

    /**
     * Show admin profile page
     */
    public function profile()
    {
        // Cek jika user sudah login via Laravel Auth
        if (Auth::check()) {
            return view('admin.profile', [
                'user' => Auth::user(),
            ]);
        }
        
        // Fallback: cek session manual
        if (Session::has('user')) {
            return view('admin.profile', [
                'user' => Session::get('user'),
            ]);
        }

        return redirect()->route('admin.login.form')
            ->with('error', 'Please login first');
    }

    /**
     * Update admin profile
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user() ?? Session::get('user');
        
        if (!$user) {
            return redirect()->route('admin.login.form');
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . $user->id,
            'phone' => 'required|string|max:20',
            'current_password' => 'nullable|string|min:6',
            'new_password' => 'nullable|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Update langsung ke database
        $userToUpdate = User::find($user->id);
        
        if (!$userToUpdate) {
            return back()->with('error', 'User not found');
        }

        $userToUpdate->name = $request->name;
        $userToUpdate->email = $request->email;
        $userToUpdate->phone = $request->phone;

        // Jika ingin ganti password
        if ($request->filled('current_password') && $request->filled('new_password')) {
            if (!Hash::check($request->current_password, $userToUpdate->password)) {
                return back()->withErrors([
                    'current_password' => 'Current password is incorrect',
                ])->withInput();
            }
            $userToUpdate->password = Hash::make($request->new_password);
        }

        $userToUpdate->save();

        // Update session data
        Session::put('user', $userToUpdate);
        
        // Jika menggunakan Laravel Auth, update juga
        if (Auth::check()) {
            Auth::setUser($userToUpdate);
        }

        return redirect()->route('admin.profile')
            ->with('success', 'Profile updated successfully');
    }

    /**
     * Logout admin
     */
    public function logout()
    {
        // Logout dari Laravel Auth
        Auth::logout();
        
        // Clear semua session
        Session::flush();

        return redirect()->route('admin.login.form')
            ->with('success', 'Logged out successfully');
    }

    /**
     * Show admin users management page
     */
    public function usersIndex(Request $request)
    {
        // Cek authorization
        $user = Auth::user() ?? Session::get('user');
        if (!$user || $user->role !== 'admin') {
            return redirect()->route('admin.login.form');
        }

        // Gunakan API request untuk data users
        $response = $this->apiRequest('get', 'admin/users', $request->all());

        if ($response['success'] && $response['data']['success']) {
            return view('admin.users.index', [
                'users' => $response['data']['data'] ?? [],
                'pagination' => $response['data']['meta'] ?? null,
                'roles' => ['admin', 'employee', 'customer'],
                'search' => $request->search,
            ]);
        }

        return view('admin.users.index', [
            'users' => [],
            'roles' => ['admin', 'employee', 'customer'],
        ]);
    }

    /**
     * Show user detail
     */
    public function showUser($id)
    {
        $response = $this->apiRequest('get', "admin/users/{$id}");

        if ($response['success'] && $response['data']['success']) {
            return view('admin.users.show', [
                'user' => $response['data']['data'],
            ]);
        }

        return redirect()->route('admin.users.index')
            ->with('error', 'User not found');
    }

    /**
 * Show edit user form
 */
public function editUser($id)
{
    $response = $this->apiRequest('get', "admin/users/{$id}");

    if ($response['success'] && $response['data']['success']) {
        return view('admin.users.edit', [
            'user' => $response['data']['data'],
            'roles' => ['admin', 'employee', 'customer'],
        ]);
    }

    return redirect()->route('admin.users.index')
        ->with('error', 'User not found');
}

    /**
     * Update user
     */
    public function updateUser(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'phone' => 'required|string|max:20',
            'role' => 'required|in:admin,employee,customer',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        $response = $this->apiRequest('put', "admin/users/{$id}", $request->all());

        if ($response['success'] && $response['data']['success']) {
            return redirect()->route('admin.users.show', $id)
                ->with('success', 'User updated successfully');
        }

        $errorMessage = $response['data']['message'] ?? 'Failed to update user';
        return back()->with('error', $errorMessage)->withInput();
    }

    /**
     * Delete user
     */
    public function deleteUser($id)
    {
        $response = $this->apiRequest('delete', "admin/users/{$id}");

        if ($response['success'] && $response['data']['success']) {
            return redirect()->route('admin.users.index')
                ->with('success', 'User deleted successfully');
        }

        $errorMessage = $response['data']['message'] ?? 'Failed to delete user';
        return back()->with('error', $errorMessage);
    }

    /**
     * Show create user form
     */
    public function createUser()
    {
        return view('admin.users.create', [
            'roles' => ['admin', 'employee', 'customer'],
        ]);
    }

    /**
     * Store new user
     */
    public function storeUser(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email',
            'phone' => 'required|string|max:20',
            'password' => 'required|string|min:6|confirmed',
            'role' => 'required|in:admin,employee,customer',
        ]);

        if ($validator->fails()) {
            return back()->withErrors($validator)->withInput();
        }

        // Create user langsung ke database
        $user = User::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'password' => Hash::make($request->password),
            'role' => $request->role,
        ]);

        return redirect()->route('admin.users.index')
            ->with('success', 'User created successfully');
    }

    /**
     * Show admin dashboard
     */
    public function dashboard()
    {
        // Cek jika sudah login
        if (!Auth::check() && !Session::has('is_admin')) {
            return redirect()->route('admin.login.form');
        }

        // Gunakan API untuk dashboard data jika ada
        $response = $this->apiRequest('get', 'admin/auth/dashboard');

        if ($response['success'] && $response['data']['success']) {
            return view('admin.dashboard', [
                'dashboardData' => $response['data']['data'],
            ]);
        }

        // Fallback ke view tanpa data
        return view('admin.dashboard', [
            'dashboardData' => [],
            'error' => $response['data']['message'] ?? 'Failed to load dashboard data'
        ]);
    }
}