<?php
// app/Http\Controllers/Web/Admin/SystemSettingController.php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class SystemSettingController extends BaseApiController
{
    /**
     * Display system settings
     */
    public function index()
    {
        $response = $this->apiRequest('get', 'admin/system-settings');
        
        return $this->handleResponse($response, 'admin.settings.index', [
            'title' => 'System Settings',
            'settingGroups' => ['general', 'payment', 'tax', 'operational', 'notification']
        ]);
    }

    /**
     * Update general settings
     */
    public function updateGeneral(Request $request)
    {
        $request->validate([
            'app_name' => 'required|string|max:100',
            'app_logo' => 'nullable|string|max:255',
            'app_description' => 'nullable|string|max:500',
            'currency' => 'required|string|size:3',
            'currency_symbol' => 'required|string|max:5',
            'timezone' => 'required|timezone',
        ]);
        
        $settings = [];
        foreach ($request->all() as $key => $value) {
            if (in_array($key, ['_token', '_method'])) continue;
            $settings[] = ['key' => $key, 'value' => $value];
        }
        
        $response = $this->apiRequest('post', 'admin/system-settings/bulk-update', [
            'settings' => $settings
        ]);
        
        return $this->handleRedirectResponse($response, 
            'admin.settings.index', 
            [],
            'General settings updated'
        );
    }

    /**
     * Update operational hours
     */
    public function updateOperationalHours(Request $request)
    {
        $days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'];
        
        $operationalHours = [];
        foreach ($days as $index => $day) {
            $operationalHours[] = [
                'day_of_week' => $index,
                'is_open' => $request->input("{$day}_open", false),
                'open_time' => $request->input("{$day}_open_time", '08:00'),
                'close_time' => $request->input("{$day}_close_time", '22:00'),
            ];
        }
        
        $response = $this->apiRequest('put', 'admin/system-settings/operational-hours', [
            'hours' => $operationalHours
        ]);
        
        return $this->handleRedirectResponse($response, 
            'admin.settings.index', 
            [],
            'Operational hours updated'
        );
    }

    /**
     * Update tax settings
     */
    public function updateTax(Request $request)
    {
        $request->validate([
            'tax_rate' => 'required|numeric|min:0|max:100',
            'service_fee' => 'required|numeric|min:0',
            'tax_inclusive' => 'boolean',
        ]);
        
        $settings = [];
        foreach ($request->all() as $key => $value) {
            if (in_array($key, ['_token', '_method'])) continue;
            $settings[] = ['key' => $key, 'value' => $value];
        }
        
        $response = $this->apiRequest('post', 'admin/system-settings/bulk-update', [
            'settings' => $settings
        ]);
        
        return $this->handleRedirectResponse($response, 
            'admin.settings.index', 
            [],
            'Tax settings updated'
        );
    }

    /**
     * Get operational hours
     */
    public function getOperationalHours()
    {
        $response = $this->apiRequest('get', 'admin/system-settings/operational-hours');
        
        return $this->jsonResponse($response);
    }
}