<?php
// app/Http\Controllers\Web/Admin/ReportController.php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class ReportController extends BaseApiController
{
    /**
     * Display sales report
     */
    public function sales(Request $request)
    {
        $response = $this->apiRequest('get', 'admin/reports/sales', $request->all());
        
        return $this->handleResponse($response, 'admin.reports.sales', [
            'title' => 'Sales Report',
            'period' => $request->period ?? 'today',
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
        ]);
    }

    /**
     * Display finance report
     */
    public function finance(Request $request)
    {
        // Combine multiple API calls for finance report
        $salesResponse = $this->apiRequest('get', 'admin/reports/sales', $request->all());
        $paymentsResponse = $this->apiRequest('get', 'admin/payments/statistics', $request->all());
        
        $viewData = [
            'title' => 'Finance Report',
            'salesData' => $salesResponse['success'] ? 
                ($salesResponse['data']['data'] ?? []) : [],
            'paymentsData' => $paymentsResponse['success'] ? 
                ($paymentsResponse['data']['data'] ?? []) : [],
            'period' => $request->period ?? 'month',
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
        ];
        
        return view('admin.reports.finance', $viewData);
    }

    /**
     * Display performance report
     */
    public function performance(Request $request)
    {
        $ordersResponse = $this->apiRequest('get', 'admin/orders/statistics', $request->all());
        $menuResponse = $this->apiRequest('get', 'admin/menus/statistics', $request->all());
        
        $viewData = [
            'title' => 'Performance Report',
            'ordersData' => $ordersResponse['success'] ? 
                ($ordersResponse['data']['data'] ?? []) : [],
            'menuData' => $menuResponse['success'] ? 
                ($menuResponse['data']['data'] ?? []) : [],
            'period' => $request->period ?? 'month',
        ];
        
        return view('admin.reports.performance', $viewData);
    }

    /**
     * Display orders by date range
     */
    public function ordersByDateRange(Request $request)
    {
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
        ]);
        
        $response = $this->apiRequest('get', 'admin/orders', [
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
            'export' => $request->export ?? false,
        ]);
        
        if ($request->export === 'pdf' || $request->export === 'excel') {
            return $this->handleExport($response, $request->export);
        }
        
        return $this->handleResponse($response, 'admin.reports.orders-date-range', [
            'title' => 'Orders Report',
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
        ]);
    }

    /**
     * Export data
     */
    private function handleExport($apiResponse, $format)
    {
        if (!$apiResponse['success'] || !$apiResponse['data']['success']) {
            return redirect()->back()->with('error', 'Failed to generate export');
        }
        
        $data = $apiResponse['data']['data'] ?? [];
        
        if ($format === 'pdf') {
            // PDF export logic
            return response()->streamDownload(function() use ($data) {
                echo "Orders Report PDF\n\n";
                // Add actual PDF generation here
            }, 'orders-report.pdf');
        }
        
        if ($format === 'excel') {
            // Excel export logic
            return response()->streamDownload(function() use ($data) {
                echo "Orders Report Excel\n\n";
                // Add actual Excel generation here
            }, 'orders-report.xlsx');
        }
        
        return redirect()->back()->with('error', 'Invalid export format');
    }

    /**
     * Get report data for AJAX
     */
    public function getReportData(Request $request)
    {
        $type = $request->type ?? 'sales';
        $period = $request->period ?? 'today';
        
        $endpoint = match($type) {
            'sales' => 'admin/reports/sales',
            'orders' => 'admin/orders/statistics',
            'payments' => 'admin/payments/statistics',
            'customers' => 'admin/reports/customers',
            default => 'admin/reports/sales'
        };
        
        $response = $this->apiRequest('get', $endpoint, [
            'period' => $period,
            'start_date' => $request->start_date,
            'end_date' => $request->end_date,
        ]);
        
        return $this->jsonResponse($response);
    }
}