<?php
// app/Http\Controllers\Web\Admin\PromoManagementController.php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class PromoManagementController extends BaseApiController
{
    public function index(Request $request)
    {
        set_time_limit(120); 
        $params = $request->only(['search', 'type', 'is_active', 'page']);
        
        $response = $this->apiRequest('get', 'promos', $params);
        
        if ($response['success']) {
            return view('admin.promos.index', [
                'promos' => $response['data']['data'] ?? [],
                'pagination' => $response['data']['meta'] ?? null,
                'types' => ['percentage', 'fixed', 'bogo'],
                'filters' => $params,
            ]);
        }
        
        return view('admin.promos.index', [
            'promos' => [],
            'pagination' => null,
            'types' => ['percentage', 'fixed', 'bogo'],
            'filters' => $params,
            'title' => 'Promo Management',
        ]);
    }

    public function create()
    {
        return view('admin.promos.create', [
            'types' => ['percentage', 'fixed', 'bogo'],
        ]);
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:percentage,fixed,bogo',
            'discount_value' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'code' => 'nullable|string|max:50|unique:promos,code',
            'min_purchase' => 'nullable|numeric|min:0',
            'max_discount' => 'nullable|numeric|min:0',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_active' => 'boolean',
        ]);
        
        $data = $request->all();
        $data['is_active'] = $request->boolean('is_active');
        
        $response = $this->apiRequest('post', 'promos', $data);
        
        if ($response['success']) {
            return redirect()->route('admin.promos.index')
                ->with('success', 'Promotion created successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to create promotion']
        )->withInput();
    }

    public function show($id)
    {
        $response = $this->apiRequest('get', "promos/{$id}");
        
        if ($response['success']) {
            return view('admin.promos.show', [
                'promo' => $response['data']['data'] ?? $response['data'],
            ]);
        }
        
        return redirect()->route('admin.promos.index')
            ->with('error', 'Promotion not found');
    }

    public function edit($id)
    {
        $response = $this->apiRequest('get', "promos/{$id}");
        
        if ($response['success']) {
            return view('admin.promos.edit', [
                'promo' => $response['data']['data'] ?? $response['data'],
                'types' => ['percentage', 'fixed', 'bogo'],
            ]);
        }
        
        return redirect()->route('admin.promos.index')
            ->with('error', 'Promotion not found');
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'type' => 'required|in:percentage,fixed,bogo',
            'discount_value' => 'required|numeric|min:0',
            'description' => 'nullable|string',
            'code' => 'nullable|string|max:50',
            'min_purchase' => 'nullable|numeric|min:0',
            'max_discount' => 'nullable|numeric|min:0',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after:start_date',
            'is_active' => 'boolean',
        ]);
        
        $data = $request->all();
        $data['is_active'] = $request->boolean('is_active');
        
        $response = $this->apiRequest('put', "promos/{$id}", $data);
        
        if ($response['success']) {
            return redirect()->route('admin.promos.index')
                ->with('success', 'Promotion updated successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to update promotion']
        )->withInput();
    }

    public function destroy($id)
    {
        $response = $this->apiRequest('delete', "promos/{$id}");
        
        if ($response['success']) {
            return redirect()->route('admin.promos.index')
                ->with('success', 'Promotion deleted successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to delete promotion']
        );
    }

    public function toggleActive($id)
    {
        $promoResponse = $this->apiRequest('get', "promos/{$id}");
        
        if ($promoResponse['success']) {
            $promo = $promoResponse['data']['data'] ?? $promoResponse['data'];
            $newStatus = !$promo['is_active'];
            
            $response = $this->apiRequest('put', "promos/{$id}", [
                'is_active' => $newStatus,
            ]);
            
            if ($response['success']) {
                $status = $newStatus ? 'activated' : 'deactivated';
                return redirect()->route('admin.promos.index')
                    ->with('success', "Promotion {$status}");
            }
        }
        
        return redirect()->route('admin.promos.index')
            ->with('error', 'Failed to update promotion status');
    }
}