<?php
// app/Http\Controllers\Web\Admin\PaymentMethodController.php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class PaymentMethodController extends BaseApiController
{
    /**
     * Display all payment methods
     */
public function index(Request $request)
{
    set_time_limit(120); 
    $params = $request->only(['search', 'is_active', 'code', 'page']);
    
    $response = $this->apiRequest('get', 'payment-methods', $params);
    
    $statsResponse = $this->apiRequest('get', 'reports/payment-methods/stats');

    if ($response['success']) {
        $stats = $statsResponse['success'] ? $statsResponse['data'] : [];
        
        return view('admin.payment-methods.index', [
            'paymentMethods' => $response['data']['data'] ?? [],
            'pagination' => $response['data']['meta'] ?? null,
            'codes' => ['cash', 'bank_transfer', 'ewallet', 'qris', 'credit_card'],
            'filters' => $params,
            'totalMethods' => $stats['total_methods'] ?? 0,
            'activeMethods' => $stats['active_methods'] ?? 0,
            'digitalMethods' => $stats['digital_methods'] ?? 0,
            'mostUsedMethod' => $stats['most_used_method'] ?? 'N/A',
        ]);
    }
    
    // Add pagination with null or empty default in error scenario
    return view('admin.payment-methods.index', [
        'paymentMethods' => [],
        'pagination' => null,  // Add this line
        'codes' => ['cash', 'bank_transfer', 'ewallet', 'qris', 'credit_card'],
        'filters' => $params,
        'totalMethods' => 0,
        'activeMethods' => 0,
        'digitalMethods' => 0,
        'mostUsedMethod' => 'N/A',
    ]);
}
    /**
     * Show create payment method form
     */
    public function create()
    {
        return view('admin.payment-methods.create', [
            'codes' => ['cash', 'bank_transfer', 'ewallet', 'qris', 'credit_card'],
        ]);
    }

    /**
     * Store new payment method
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:payment_methods,name',
            'code' => 'required|in:cash,bank_transfer,ewallet,qris,credit_card',
            'account_number' => 'nullable|string|max:100',
            'account_name' => 'nullable|string|max:255',
            'instructions' => 'nullable|string|max:1000',
            'is_active' => 'boolean',
        ]);
        
        $data = $request->all();
        $data['is_active'] = $request->boolean('is_active');
        
        $response = $this->apiRequest('post', 'payment-methods', $data);
        
        if ($response['success']) {
            return redirect()->route('admin.payment-methods.index')
                ->with('success', 'Payment method created successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to create payment method']
        )->withInput();
    }

    /**
     * Show payment method details
     */
    public function show($id)
    {
        $response = $this->apiRequest('get', "payment-methods/{$id}");
        
        if ($response['success']) {
        $paymentMethod = $response['data']['data'] ?? $response['data'];
        
        // Get recent payments for this method
        $paymentsResponse = $this->apiRequest('get', 'payments', [
            'payment_method_id' => $id,
            'limit' => 10,
        ]);
        
        // Get statistics
        $statsResponse = $this->apiRequest('get', "payment-methods/{$id}/statistics");
        
        return view('admin.payment-methods.show', [
            'paymentMethod' => $paymentMethod,
            'recentPayments' => $paymentsResponse['success'] ? 
                ($paymentsResponse['data']['data'] ?? []) : [],
            'successRate' => $statsResponse['success'] ? 
                ($statsResponse['data']['success_rate'] ?? 100) : 100,
            'averageAmount' => $statsResponse['success'] ? 
                ($statsResponse['data']['average_amount'] ?? 0) : 0,
            'totalRevenue' => $statsResponse['success'] ? 
                ($statsResponse['data']['total_revenue'] ?? 0) : 0,
        ]);        }
        
        return redirect()->route('admin.payment-methods.index')
            ->with('error', 'Payment method not found');
    }

    /**
     * Show edit payment method form
     */
    public function edit($id)
    {
        $response = $this->apiRequest('get', "payment-methods/{$id}");
        
        if ($response['success']) {
            return view('admin.payment-methods.edit', [
                'paymentMethod' => $response['data']['data'] ?? $response['data'],
                'codes' => ['cash', 'bank_transfer', 'ewallet', 'qris', 'credit_card'],
            ]);
        }
        
        return redirect()->route('admin.payment-methods.index')
            ->with('error', 'Payment method not found');
    }

    /**
     * Update payment method
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|in:cash,bank_transfer,ewallet,qris,credit_card',
            'account_number' => 'nullable|string|max:100',
            'account_name' => 'nullable|string|max:255',
            'instructions' => 'nullable|string|max:1000',
            'is_active' => 'boolean',
        ]);
        
        $data = $request->all();
        $data['is_active'] = $request->boolean('is_active');
        
        $response = $this->apiRequest('put', "payment-methods/{$id}", $data);
        
        if ($response['success']) {
            return redirect()->route('admin.payment-methods.index')
                ->with('success', 'Payment method updated successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to update payment method']
        )->withInput();
    }

    /**
     * Delete payment method
     */
    public function destroy($id)
    {
        $response = $this->apiRequest('delete', "payment-methods/{$id}");
        
        if ($response['success']) {
            return redirect()->route('admin.payment-methods.index')
                ->with('success', 'Payment method deleted successfully');
        }
        
        return back()->withErrors(
            $response['data']['errors'] ?? ['error' => 'Failed to delete payment method']
        );
    }

    /**
     * Toggle payment method active status
     */
    public function toggleActive($id)
    {
        $methodResponse = $this->apiRequest('get', "payment-methods/{$id}");
        
        if ($methodResponse['success']) {
            $method = $methodResponse['data']['data'] ?? $methodResponse['data'];
            $newStatus = !$method['is_active'];
            
            $response = $this->apiRequest('put', "payment-methods/{$id}", [
                'is_active' => $newStatus,
            ]);
            
            if ($response['success']) {
                $status = $newStatus ? 'activated' : 'deactivated';
                return redirect()->route('admin.payment-methods.index')
                    ->with('success', "Payment method {$status}");
            }
        }
        
        return redirect()->route('admin.payment-methods.index')
            ->with('error', 'Failed to update payment method status');
    }

    /**
     * Get payment method usage statistics
     */
    public function usageStatistics($id)
    {
        $response = $this->apiRequest('get', "payment-methods/{$id}/statistics");
        
        if ($response['success']) {
            return view('admin.payment-methods.statistics', [
                'paymentMethod' => $response['data']['payment_method'] ?? null,
                'statistics' => $response['data']['statistics'] ?? [],
                'chartData' => $response['data']['chart_data'] ?? [],
            ]);
        }
        
        return redirect()->route('admin.payment-methods.show', $id)
            ->with('error', 'Failed to load usage statistics');
    }
}