<?php
// app/Http/Controllers/Web/Admin/CareerManagementController.php

namespace App\Http\Controllers\Web\Admin;

use App\Http\Controllers\Web\Api\BaseApiController;
use Illuminate\Http\Request;

class CareerManagementController extends BaseApiController
{
    /**
     * Display career listings
     */
    public function index(Request $request)
    {
        $response = $this->apiRequest('get', 'admin/careers', $request->all());
        
        return $this->handleResponse($response, 'admin.careers.index', [
            'title' => 'Career Management',
            'filters' => $request->all(),
        ]);
    }

    /**
     * Show create career form
     */
    public function create()
    {
        return view('admin.careers.create', [
            'title' => 'Create Career',
            'employmentTypes' => ['full_time', 'part_time', 'contract', 'internship']
        ]);
    }

    /**
     * Store new career
     */
    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'department' => 'required|string|max:100',
            'location' => 'required|string|max:100',
            'description' => 'required|string',
            'requirements' => 'required|string',
            'benefits' => 'nullable|string',
            'employment_type' => 'required|in:full_time,part_time,contract,internship',
            'salary_range' => 'nullable|string|max:100',
            'application_deadline' => 'required|date',
        ]);
        
        $response = $this->apiRequest('post', 'admin/careers', $request->all());
        
        return $this->handleRedirectResponse($response, 
            'admin.careers.index', 
            [],
            'Career created successfully'
        );
    }

    /**
     * Display career details
     */
    public function show($id)
    {
        $response = $this->apiRequest('get', "admin/careers/{$id}");
        
        return $this->handleResponse($response, 'admin.careers.show', [
            'title' => 'Career Details'
        ]);
    }

    /**
     * Show edit career form
     */
    public function edit($id)
    {
        $response = $this->apiRequest('get', "admin/careers/{$id}");
        
        if ($response['success'] && $response['data']['success']) {
            return view('admin.careers.edit', [
                'title' => 'Edit Career',
                'career' => $response['data']['data'],
                'employmentTypes' => ['full_time', 'part_time', 'contract', 'internship']
            ]);
        }
        
        return redirect()->route('admin.careers.index')
            ->with('error', 'Career not found');
    }

    /**
     * Update career
     */
    public function update(Request $request, $id)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'department' => 'required|string|max:100',
            'location' => 'required|string|max:100',
            'description' => 'required|string',
            'requirements' => 'required|string',
            'benefits' => 'nullable|string',
            'employment_type' => 'required|in:full_time,part_time,contract,internship',
            'salary_range' => 'nullable|string|max:100',
            'application_deadline' => 'required|date',
        ]);
        
        $response = $this->apiRequest('put', "admin/careers/{$id}", $request->all());
        
        return $this->handleRedirectResponse($response, 
            'admin.careers.show', 
            ['id' => $id],
            'Career updated successfully'
        );
    }

    /**
     * Delete career
     */
    public function destroy($id)
    {
        $response = $this->apiRequest('delete', "admin/careers/{$id}");
        
        return $this->handleRedirectResponse($response, 
            'admin.careers.index', 
            [],
            'Career deleted successfully'
        );
    }

    /**
     * Display career applications
     */
    public function applications(Request $request)
    {
        $response = $this->apiRequest('get', 'admin/careers/applications', $request->all());
        
        return $this->handleResponse($response, 'admin.career-applications.index', [
            'title' => 'Career Applications',
            'filters' => $request->all(),
            'statuses' => ['pending', 'reviewed', 'interviewed', 'accepted', 'rejected']
        ]);
    }

    /**
     * Show application details
     */
    public function showApplication($id)
    {
        $response = $this->apiRequest('get', "admin/careers/applications/{$id}");
        
        return $this->handleResponse($response, 'admin.career-applications.show', [
            'title' => 'Application Details',
            'statuses' => ['pending', 'reviewed', 'interviewed', 'accepted', 'rejected']
        ]);
    }

    /**
     * Update application status
     */
    public function updateApplicationStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:pending,reviewed,interviewed,accepted,rejected',
            'notes' => 'nullable|string|max:500',
        ]);
        
        $response = $this->apiRequest('put', "admin/careers/applications/{$id}/status", [
            'status' => $request->status,
            'notes' => $request->notes,
        ]);
        
        return $this->handleRedirectResponse($response, 
            'admin.career-applications.show', 
            ['id' => $id],
            'Application status updated'
        );
    }
}