<?php
// app/Http/Controllers/Api/Employee/OrderController.php
namespace App\Http\Controllers\Api\Employee;

use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Employee\UpdateOrderStatusRequest;
use App\Http\Requests\Api\Employee\FilterOrdersRequest;
use App\Http\Resources\Api\Employee\OrderResource;
use App\Models\Order;
use App\Models\OrderItem;
use App\Traits\ApiResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class OrderController extends Controller
{
    use ApiResponse;

    /**
     * Get orders with filters
     * GET /api/employee/orders
     */
    public function index(FilterOrdersRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();
            
            $orders = Order::with(['customer:id,name,phone', 'items.menu', 'payments'])
                ->when($validated['status'] ?? null, function ($query, $status) {
                    return $query->where('status', $status);
                })
                ->when($validated['order_type'] ?? null, function ($query, $type) {
                    return $query->where('order_type', $type);
                })
                ->when($validated['date_from'] ?? null, function ($query) use ($validated) {
                    return $query->whereDate('created_at', '>=', $validated['date_from']);
                })
                ->when($validated['date_to'] ?? null, function ($query) use ($validated) {
                    return $query->whereDate('created_at', '<=', $validated['date_to']);
                })
                ->when($validated['search'] ?? null, function ($query, $search) {
                    return $query->where(function ($q) use ($search) {
                        $q->where('order_number', 'like', "%{$search}%")
                          ->orWhereHas('customer', function ($q2) use ($search) {
                              $q2->where('name', 'like', "%{$search}%")
                                 ->orWhere('phone', 'like', "%{$search}%");
                          });
                    });
                })
                ->orderBy('created_at', 'desc')
                ->paginate($validated['per_page'] ?? 20);

            // Statistics for the filtered results
            $stats = [
                'total_orders' => $orders->total(),
                'total_amount' => Order::when($validated['status'] ?? null, function ($query, $status) {
                                        return $query->where('status', $status);
                                    })
                                    ->when($validated['date_from'] ?? null, function ($query) use ($validated) {
                                        return $query->whereDate('created_at', '>=', $validated['date_from']);
                                    })
                                    ->when($validated['date_to'] ?? null, function ($query) use ($validated) {
                                        return $query->whereDate('created_at', '<=', $validated['date_to']);
                                    })
                                    ->sum('final_amount'),
                'average_order_value' => $orders->total() > 0 
                    ? Order::when($validated['status'] ?? null, function ($query, $status) {
                            return $query->where('status', $status);
                        })
                        ->when($validated['date_from'] ?? null, function ($query) use ($validated) {
                            return $query->whereDate('created_at', '>=', $validated['date_from']);
                        })
                        ->when($validated['date_to'] ?? null, function ($query) use ($validated) {
                            return $query->whereDate('created_at', '<=', $validated['date_to']);
                        })
                        ->avg('final_amount')
                    : 0,
            ];

            return $this->successResponse([
                'orders' => OrderResource::collection($orders),
                'pagination' => [
                    'total' => $orders->total(),
                    'per_page' => $orders->perPage(),
                    'current_page' => $orders->currentPage(),
                    'last_page' => $orders->lastPage(),
                    'from' => $orders->firstItem(),
                    'to' => $orders->lastItem(),
                ],
                'statistics' => $stats,
                'filters' => [
                    'status' => $validated['status'] ?? null,
                    'order_type' => $validated['order_type'] ?? null,
                    'date_from' => $validated['date_from'] ?? null,
                    'date_to' => $validated['date_to'] ?? null,
                    'search' => $validated['search'] ?? null,
                ]
            ], 'Daftar pesanan berhasil diambil');

        } catch (\Exception $e) {
            return $this->errorResponse('Gagal mengambil daftar pesanan: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get order detail
     * GET /api/employee/orders/{id}
     */
    public function show(Request $request, $id): JsonResponse
    {
        try {
            $order = Order::with([
                'customer:id,name,email,phone',
                'items.menu.category',
                'payments.paymentMethod',
                'employee:id,name'
            ])->find($id);

            if (!$order) {
                return $this->errorResponse('Pesanan tidak ditemukan', 404);
            }

            return $this->successResponse(
                new OrderResource($order),
                'Detail pesanan berhasil diambil'
            );

        } catch (\Exception $e) {
            return $this->errorResponse('Gagal mengambil detail pesanan: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Update order status
     * PUT /api/employee/orders/{id}/status
     */
    public function updateStatus(UpdateOrderStatusRequest $request, $id): JsonResponse
    {
        DB::beginTransaction();
        
        try {
            $order = Order::find($id);
            
            if (!$order) {
                return $this->errorResponse('Pesanan tidak ditemukan', 404);
            }

            $validated = $request->validated();
            $oldStatus = $order->status;
            $newStatus = $validated['status'];
            
            // Validate status transition
            if (!$this->isValidStatusTransition($oldStatus, $newStatus)) {
                return $this->errorResponse(
                    "Tidak dapat mengubah status dari {$this->getFormattedStatus($oldStatus)} ke {$this->getFormattedStatus($newStatus)}",
                    400
                );
            }

            // Update order status
            $order->update([
                'status' => $newStatus,
                'employee_id' => $request->user()->id,
                'completed_at' => $newStatus === 'completed' ? now() : $order->completed_at,
                'notes' => $order->notes . "\n[Status changed: {$oldStatus} → {$newStatus} by Employee {$request->user()->id}]"
            ]);

            DB::commit();

            return $this->successResponse(
                new OrderResource($order->fresh(['customer', 'items', 'payments'])),
                "✅ Status pesanan berhasil diubah menjadi: {$this->getFormattedStatus($newStatus)}"
            );

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Gagal mengubah status pesanan: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Assign order to employee
     * POST /api/employee/orders/{id}/assign
     */
    public function assignOrder(Request $request, $id): JsonResponse
    {
        DB::beginTransaction();
        
        try {
            $order = Order::find($id);
            
            if (!$order) {
                return $this->errorResponse('Pesanan tidak ditemukan', 404);
            }

            $employee = $request->user();
            
            $order->update([
                'employee_id' => $employee->id,
            ]);

            DB::commit();

            return $this->successResponse(
                new OrderResource($order->fresh(['employee', 'customer'])),
                "✅ Pesanan berhasil ditugaskan kepada {$employee->name}"
            );

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Gagal menugaskan pesanan: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get today's orders summary
     * GET /api/employee/orders/today
     */
    public function todaysOrders(Request $request): JsonResponse
    {
        try {
            $today = now()->format('Y-m-d');
            
            $orders = Order::with(['customer:id,name', 'items.menu'])
                ->whereDate('created_at', $today)
                ->orderBy('created_at', 'desc')
                ->get();

            // Group by status
            $groupedOrders = [
                'pending' => $orders->where('status', 'pending')->values(),
                'confirmed' => $orders->where('status', 'confirmed')->values(),
                'preparing' => $orders->where('status', 'preparing')->values(),
                'ready' => $orders->where('status', 'ready')->values(),
                'completed' => $orders->where('status', 'completed')->values(),
            ];

            // Statistics
            $stats = [
                'total' => $orders->count(),
                'total_revenue' => $orders->where('status', 'completed')->sum('final_amount'),
                'average_preparation_time' => $this->calculateAveragePreparationTime($orders),
                'most_ordered_menu' => $this->getMostOrderedMenu($orders),
            ];

            return $this->successResponse([
                'orders_by_status' => [
                    'pending' => OrderResource::collection($groupedOrders['pending']),
                    'confirmed' => OrderResource::collection($groupedOrders['confirmed']),
                    'preparing' => OrderResource::collection($groupedOrders['preparing']),
                    'ready' => OrderResource::collection($groupedOrders['ready']),
                    'completed' => OrderResource::collection($groupedOrders['completed']),
                ],
                'counts' => [
                    'pending' => $groupedOrders['pending']->count(),
                    'confirmed' => $groupedOrders['confirmed']->count(),
                    'preparing' => $groupedOrders['preparing']->count(),
                    'ready' => $groupedOrders['ready']->count(),
                    'completed' => $groupedOrders['completed']->count(),
                ],
                'statistics' => $stats,
                'date' => $today,
            ], 'Ringkasan pesanan hari ini berhasil diambil');

        } catch (\Exception $e) {
            return $this->errorResponse('Gagal mengambil pesanan hari ini: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get orders by status
     * GET /api/employee/orders/status/{status}
     */
    public function getOrdersByStatus(Request $request, $status): JsonResponse
    {
        try {
            if (!in_array($status, ['pending', 'confirmed', 'preparing', 'ready', 'completed', 'cancelled'])) {
                return $this->errorResponse('Status tidak valid', 400);
            }

            $orders = Order::with(['customer:id,name,phone', 'items.menu'])
                ->where('status', $status)
                ->orderBy('created_at', $status === 'ready' ? 'asc' : 'desc') // Ready orders by oldest first
                ->paginate($request->per_page ?? 20);

            return $this->successResponse([
                'orders' => OrderResource::collection($orders),
                'pagination' => [
                    'total' => $orders->total(),
                    'per_page' => $orders->perPage(),
                    'current_page' => $orders->currentPage(),
                    'last_page' => $orders->lastPage(),
                ],
                'status_info' => [
                    'status' => $status,
                    'formatted_status' => $this->getFormattedStatus($status),
                    'total_count' => Order::where('status', $status)->count(),
                    'instructions' => $this->getStatusInstructions($status),
                ]
            ], "Daftar pesanan dengan status: {$this->getFormattedStatus($status)}");

        } catch (\Exception $e) {
            return $this->errorResponse('Gagal mengambil pesanan: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Generate receipt/invoice (JSON data for printing)
     * GET /api/employee/orders/{id}/receipt
     */
    public function generateReceipt(Request $request, $id): JsonResponse
    {
        try {
            $order = Order::with([
                'customer:id,name,phone',
                'items.menu',
                'payments.paymentMethod',
                'employee:id,name'
            ])->find($id);

            if (!$order) {
                return $this->errorResponse('Pesanan tidak ditemukan', 404);
            }

            $receiptData = [
                'receipt_number' => 'INV-' . date('Ymd') . '-' . str_pad($order->id, 6, '0', STR_PAD_LEFT),
                'order_number' => $order->order_number,
                'date' => $order->created_at->format('d/m/Y H:i:s'),
                'customer' => [
                    'name' => $order->customer->name,
                    'phone' => $order->customer->phone,
                ],
                'order_type' => $this->getFormattedOrderType($order->order_type),
                'table_number' => $order->table_number,
                'items' => $order->items->map(function ($item) {
                    return [
                        'name' => $item->menu->name,
                        'quantity' => $item->quantity,
                        'price' => (float) $item->unit_price,
                        'formatted_price' => 'Rp ' . number_format($item->unit_price, 0, ',', '.'),
                        'total' => (float) $item->subtotal,
                        'formatted_total' => 'Rp ' . number_format($item->subtotal, 0, ',', '.'),
                    ];
                }),
                'payment' => $order->payments->first() ? [
                    'method' => $order->payments->first()->paymentMethod->name,
                    'status' => $this->getFormattedPaymentStatus($order->payments->first()->status),
                    'reference' => $order->payments->first()->reference_number,
                ] : null,
                'summary' => [
                    'subtotal' => (float) $order->total_amount,
                    'tax' => (float) $order->tax_amount,
                    'service_fee' => (float) $order->service_fee,
                    'discount' => (float) $order->discount_amount,
                    'total' => (float) $order->final_amount,
                ],
                'formatted_summary' => [
                    'subtotal' => 'Rp ' . number_format($order->total_amount, 0, ',', '.'),
                    'tax' => 'Rp ' . number_format($order->tax_amount, 0, ',', '.'),
                    'service_fee' => 'Rp ' . number_format($order->service_fee, 0, ',', '.'),
                    'discount' => 'Rp ' . number_format($order->discount_amount, 0, ',', '.'),
                    'total' => 'Rp ' . number_format($order->final_amount, 0, ',', '.'),
                ],
                'restaurant_info' => [
                    'name' => 'Chingu Bite',
                    'address' => 'Jl. Korea Selatan No. 123, Jakarta',
                    'phone' => '(021) 1234-5678',
                    'website' => 'www.chingubite.com',
                ],
                'footer_message' => 'Terima kasih telah berkunjung!',
                'print_timestamp' => now()->format('Y-m-d H:i:s'),
            ];

            return $this->successResponse(
                $receiptData,
                'Data struk berhasil diambil'
            );

        } catch (\Exception $e) {
            return $this->errorResponse('Gagal membuat struk: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Get order statistics for charts
     * GET /api/employee/orders/statistics
     */
    public function statistics(Request $request): JsonResponse
    {
        try {
            $validated = $request->validate([
                'period' => 'nullable|in:today,week,month,year',
                'date_from' => 'nullable|date',
                'date_to' => 'nullable|date',
            ]);

            $period = $validated['period'] ?? 'today';
            $dateFrom = $validated['date_from'] ?? null;
            $dateTo = $validated['date_to'] ?? null;

            // Determine date range
            if ($dateFrom && $dateTo) {
                $startDate = $dateFrom;
                $endDate = $dateTo;
            } else {
                switch ($period) {
                    case 'week':
                        $startDate = now()->startOfWeek()->format('Y-m-d');
                        $endDate = now()->endOfWeek()->format('Y-m-d');
                        break;
                    case 'month':
                        $startDate = now()->startOfMonth()->format('Y-m-d');
                        $endDate = now()->endOfMonth()->format('Y-m-d');
                        break;
                    case 'year':
                        $startDate = now()->startOfYear()->format('Y-m-d');
                        $endDate = now()->endOfYear()->format('Y-m-d');
                        break;
                    default: // today
                        $startDate = now()->format('Y-m-d');
                        $endDate = now()->format('Y-m-d');
                }
            }

            // Get daily order count
            $dailyOrders = Order::selectRaw('DATE(created_at) as date, COUNT(*) as count')
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->groupBy('date')
                ->orderBy('date')
                ->get()
                ->map(function ($item) {
                    return [
                        'date' => $item->date,
                        'count' => $item->count,
                    ];
                });

            // Get status distribution
            $statusDistribution = Order::selectRaw('status, COUNT(*) as count')
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->groupBy('status')
                ->get()
                ->mapWithKeys(function ($item) {
                    return [$item->status => $item->count];
                });

            // Get order type distribution
            $typeDistribution = Order::selectRaw('order_type, COUNT(*) as count')
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->groupBy('order_type')
                ->get()
                ->mapWithKeys(function ($item) {
                    return [$item->order_type => $item->count];
                });

            // Get revenue by day
            $dailyRevenue = Order::selectRaw('DATE(created_at) as date, SUM(final_amount) as revenue')
                ->whereDate('created_at', '>=', $startDate)
                ->whereDate('created_at', '<=', $endDate)
                ->where('status', 'completed')
                ->groupBy('date')
                ->orderBy('date')
                ->get()
                ->map(function ($item) {
                    return [
                        'date' => $item->date,
                        'revenue' => (float) $item->revenue,
                    ];
                });

            // Top selling menus
            $topMenus = OrderItem::selectRaw('menu_id, menus.name, SUM(quantity) as total_quantity')
                ->join('menus', 'order_items.menu_id', '=', 'menus.id')
                ->join('orders', 'order_items.order_id', '=', 'orders.id')
                ->whereDate('orders.created_at', '>=', $startDate)
                ->whereDate('orders.created_at', '<=', $endDate)
                ->groupBy('menu_id', 'menus.name')
                ->orderBy('total_quantity', 'desc')
                ->limit(10)
                ->get()
                ->map(function ($item) {
                    return [
                        'menu_id' => $item->menu_id,
                        'name' => $item->name,
                        'quantity' => $item->total_quantity,
                    ];
                });

            return $this->successResponse([
                'period' => [
                    'start_date' => $startDate,
                    'end_date' => $endDate,
                    'period_name' => $period,
                ],
                'daily_orders' => $dailyOrders,
                'daily_revenue' => $dailyRevenue,
                'status_distribution' => $statusDistribution,
                'type_distribution' => $typeDistribution,
                'top_selling_menus' => $topMenus,
                'totals' => [
                    'total_orders' => Order::whereDate('created_at', '>=', $startDate)
                                         ->whereDate('created_at', '<=', $endDate)
                                         ->count(),
                    'total_revenue' => Order::whereDate('created_at', '>=', $startDate)
                                          ->whereDate('created_at', '<=', $endDate)
                                          ->where('status', 'completed')
                                          ->sum('final_amount'),
                    'average_order_value' => Order::whereDate('created_at', '>=', $startDate)
                                                ->whereDate('created_at', '<=', $endDate)
                                                ->where('status', 'completed')
                                                ->avg('final_amount'),
                ],
            ], 'Statistik pesanan berhasil diambil');

        } catch (\Exception $e) {
            return $this->errorResponse('Gagal mengambil statistik: ' . $e->getMessage(), 500);
        }
    }

    /**
     * Cancel order (employee can cancel in certain statuses)
     * POST /api/employee/orders/{id}/cancel
     */
    public function cancelOrder(Request $request, $id): JsonResponse
    {
        DB::beginTransaction();
        
        try {
            $order = Order::with(['items.menu'])->find($id);
            
            if (!$order) {
                return $this->errorResponse('Pesanan tidak ditemukan', 404);
            }

            // Check if order can be cancelled by employee
            if (!in_array($order->status, ['pending', 'confirmed', 'preparing'])) {
                return $this->errorResponse(
                    "Pesanan dengan status {$this->getFormattedStatus($order->status)} tidak dapat dibatalkan",
                    400
                );
            }

            $oldStatus = $order->status;
            $employee = $request->user();

            // Update order status
            $order->update([
                'status' => 'cancelled',
                'employee_id' => $employee->id,
                'notes' => $order->notes . "\n[Dibatalkan oleh karyawan: {$employee->name} - Status sebelumnya: {$oldStatus}]"
            ]);

            // Restore menu stock
            foreach ($order->items as $item) {
                if ($item->menu) {
                    $item->menu->stock += $item->quantity;
                    if ($item->menu->stock > 0 && !$item->menu->is_available) {
                        $item->menu->is_available = true;
                    }
                    $item->menu->save();
                }
            }

            // Update payment status if exists
            $payment = $order->payments()->first();
            if ($payment && $payment->status === 'waiting_confirmation') {
                $payment->update([
                    'status' => 'refunded',
                    'confirmed_by' => $employee->id,
                    'confirmed_at' => now(),
                    'notes' => $payment->notes . "\n[Dibatalkan bersama pesanan]"
                ]);
            }

            DB::commit();

            return $this->successResponse(
                new OrderResource($order->fresh(['customer', 'items', 'payments'])),
                "✅ Pesanan berhasil dibatalkan. Stok menu telah dikembalikan."
            );

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Gagal membatalkan pesanan: ' . $e->getMessage(), 500);
        }
    }

    // ==================== HELPER METHODS ====================

    /**
     * Check if status transition is valid
     */
    private function isValidStatusTransition($from, $to): bool
    {
        $validTransitions = [
            'pending' => ['confirmed', 'cancelled'],
            'confirmed' => ['preparing', 'cancelled'],
            'preparing' => ['ready', 'cancelled'],
            'ready' => ['completed'],
            'completed' => [],
            'cancelled' => [],
        ];

        return in_array($to, $validTransitions[$from] ?? []);
    }

    /**
     * Get formatted status text
     */
    private function getFormattedStatus($status): string
    {
        $statuses = [
            'pending' => 'Menunggu Konfirmasi',
            'confirmed' => 'Dikonfirmasi',
            'preparing' => 'Sedang Disiapkan',
            'ready' => 'Siap Diambil',
            'completed' => 'Selesai',
            'cancelled' => 'Dibatalkan',
        ];

        return $statuses[$status] ?? $status;
    }

    /**
     * Get formatted order type
     */
    private function getFormattedOrderType($type): string
    {
        $types = [
            'dine_in' => 'Makan di Tempat',
            'takeaway' => 'Bawa Pulang',
            'delivery' => 'Delivery',
        ];

        return $types[$type] ?? $type;
    }

    /**
     * Get formatted payment status
     */
    private function getFormattedPaymentStatus($status): string
    {
        $statuses = [
            'pending' => 'Menunggu',
            'waiting_confirmation' => 'Menunggu Konfirmasi',
            'confirmed' => 'Terkonfirmasi',
            'failed' => 'Gagal',
            'refunded' => 'Dikembalikan',
        ];

        return $statuses[$status] ?? $status;
    }

    /**
     * Calculate average preparation time
     */
    private function calculateAveragePreparationTime($orders): ?string
    {
        $completedOrders = $orders->where('status', 'completed')
                                 ->whereNotNull('completed_at');
        
        if ($completedOrders->isEmpty()) {
            return null;
        }

        $totalMinutes = $completedOrders->sum(function ($order) {
            return $order->created_at->diffInMinutes($order->completed_at);
        });

        $averageMinutes = $totalMinutes / $completedOrders->count();
        
        if ($averageMinutes < 60) {
            return round($averageMinutes) . ' menit';
        } else {
            $hours = floor($averageMinutes / 60);
            $minutes = round($averageMinutes % 60);
            return $hours . ' jam ' . $minutes . ' menit';
        }
    }

    /**
     * Get most ordered menu
     */
    private function getMostOrderedMenu($orders): ?array
    {
        $menuCounts = [];
        
        foreach ($orders as $order) {
            foreach ($order->items as $item) {
                $menuId = $item->menu_id;
                $menuName = $item->menu->name ?? 'Unknown';
                $quantity = $item->quantity;
                
                if (!isset($menuCounts[$menuId])) {
                    $menuCounts[$menuId] = [
                        'name' => $menuName,
                        'quantity' => 0,
                    ];
                }
                
                $menuCounts[$menuId]['quantity'] += $quantity;
            }
        }
        
        if (empty($menuCounts)) {
            return null;
        }
        
        uasort($menuCounts, function ($a, $b) {
            return $b['quantity'] <=> $a['quantity'];
        });
        
        $topMenu = reset($menuCounts);
        
        return [
            'name' => $topMenu['name'],
            'quantity' => $topMenu['quantity'],
        ];
    }

    /**
     * Get status instructions
     */
    private function getStatusInstructions($status): array
    {
        $instructions = [
            'pending' => [
                'Konfirmasi pesanan setelah pembayaran valid',
                'Periksa ketersediaan menu',
                'Hubungi customer jika ada masalah',
            ],
            'confirmed' => [
                'Serahkan ke dapur untuk diproses',
                'Periksa catatan khusus dari customer',
                'Pastikan semua item tersedia',
            ],
            'preparing' => [
                'Monitor waktu penyiapan',
                'Pastikan kualitas makanan',
                'Siapkan packaging untuk takeaway/delivery',
            ],
            'ready' => [
                'Panggil customer untuk dine-in',
                'Siapkan pesanan untuk takeaway',
                'Hubungi driver untuk delivery',
            ],
            'completed' => [
                'Bersihkan meja untuk dine-in',
                'Update sistem pembayaran',
                'Arsipkan data pesanan',
            ],
            'cancelled' => [
                'Kembalikan stok menu',
                'Update status pembayaran',
                'Hubungi customer jika perlu',
            ],
        ];

        return $instructions[$status] ?? ['Tidak ada instruksi khusus'];
    }

    /**
     * Quick update multiple orders status
     * POST /api/employee/orders/bulk-status
     */
    public function bulkUpdateStatus(Request $request): JsonResponse
    {
        $request->validate([
            'order_ids' => 'required|array|min:1',
            'order_ids.*' => 'integer|exists:orders,id',
            'status' => 'required|in:confirmed,preparing,ready,completed,cancelled',
            'notes' => 'nullable|string|max:500',
        ]);

        DB::beginTransaction();
        
        try {
            $employee = $request->user();
            $status = $request->status;
            $notes = $request->notes;
            $updatedCount = 0;
            $failedCount = 0;
            $results = [];

            foreach ($request->order_ids as $orderId) {
                $order = Order::find($orderId);
                
                if (!$order) {
                    $failedCount++;
                    $results[] = [
                        'order_id' => $orderId,
                        'success' => false,
                        'message' => 'Order not found',
                    ];
                    continue;
                }

                // Validate status transition
                if (!$this->isValidStatusTransition($order->status, $status)) {
                    $failedCount++;
                    $results[] = [
                        'order_id' => $orderId,
                        'order_number' => $order->order_number,
                        'success' => false,
                        'message' => "Cannot change status from {$this->getFormattedStatus($order->status)} to {$this->getFormattedStatus($status)}",
                    ];
                    continue;
                }

                // Update order
                $order->update([
                    'status' => $status,
                    'employee_id' => $employee->id,
                    'completed_at' => $status === 'completed' ? now() : $order->completed_at,
                    'notes' => $order->notes . "\n[Bulk update: {$order->status} → {$status} by {$employee->name}" . ($notes ? " - {$notes}" : "") . "]"
                ]);

                $updatedCount++;
                $results[] = [
                    'order_id' => $orderId,
                    'order_number' => $order->order_number,
                    'success' => true,
                    'message' => 'Status updated successfully',
                ];
            }

            DB::commit();

            return $this->successResponse([
                'results' => $results,
                'summary' => [
                    'total_processed' => count($request->order_ids),
                    'updated' => $updatedCount,
                    'failed' => $failedCount,
                    'success_rate' => count($request->order_ids) > 0 
                        ? round(($updatedCount / count($request->order_ids)) * 100, 1) 
                        : 0,
                ],
            ], "Bulk update completed: {$updatedCount} updated, {$failedCount} failed");

        } catch (\Exception $e) {
            DB::rollBack();
            return $this->errorResponse('Gagal melakukan bulk update: ' . $e->getMessage(), 500);
        }
    }
}