<?php
namespace App\Http\Controllers\Api\Customer;

use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Customer\MenuFilterRequest;
use App\Http\Requests\Api\Customer\PromoCheckRequest;
use App\Http\Resources\Api\Customer\MenuResource;
use App\Http\Resources\Api\Customer\CategoryResource;
use App\Http\Resources\Api\Customer\PromoResource;
use App\Models\Menu;
use App\Models\Category;
use App\Models\Promo;
use App\Traits\ApiResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;

class MenuController extends Controller
{
    use ApiResponse;

    /**
     * Get all categories
     * GET /api/customer/menu/categories
     */
    public function getCategories(): JsonResponse
    {
        // Cache categories for 1 hour
        $categories = Cache::remember('categories_active', 3600, function () {
            return Category::active()
                ->withCount(['menus' => function ($query) {
                    $query->available();
                }])
                ->orderBy('name')
                ->get();
        });
        
        return $this->successResponse(
            CategoryResource::collection($categories),
            'Kategori berhasil diambil'
        );
    }

    /**
     * Get all menus with filters
     * GET /api/customer/menu
     */
    public function index(MenuFilterRequest $request): JsonResponse
    {
        $validated = $request->validated();
        
        $query = Menu::with(['category'])
            ->available()
            ->when(isset($validated['category_id']), function ($q) use ($validated) {
                return $q->byCategory($validated['category_id']);
            })
            ->when(isset($validated['search']), function ($q) use ($validated) {
                return $q->search($validated['search'], ['name', 'description']);
            })
            ->when(isset($validated['special']) && $validated['special'], function ($q) {
                return $q->special();
            })
            ->when(isset($validated['best_seller']) && $validated['best_seller'], function ($q) {
                return $q->bestSeller();
            })
            ->when(isset($validated['min_price']), function ($q) use ($validated) {
                return $q->where('price', '>=', $validated['min_price']);
            })
            ->when(isset($validated['max_price']), function ($q) use ($validated) {
                return $q->where('price', '<=', $validated['max_price']);
            });

        // Apply sorting
        if (isset($validated['sort_by'])) {
            switch ($validated['sort_by']) {
                case 'price_asc':
                    $query->orderBy('price', 'asc');
                    break;
                case 'price_desc':
                    $query->orderBy('price', 'desc');
                    break;
                case 'name':
                    $query->orderBy('name', 'asc');
                    break;
                case 'latest':
                    $query->latest();
                    break;
                default:
                    $query->latest();
            }
        } else {
            $query->latest();
        }

        $perPage = $validated['per_page'] ?? 15;
        $menus = $query->paginate($perPage);

        return $this->successResponse([
            'menus' => MenuResource::collection($menus),
            'pagination' => [
                'total' => $menus->total(),
                'per_page' => $menus->perPage(),
                'current_page' => $menus->currentPage(),
                'last_page' => $menus->lastPage(),
                'from' => $menus->firstItem(),
                'to' => $menus->lastItem(),
            ],
            'filters' => [
                'search' => $validated['search'] ?? null,
                'category_id' => $validated['category_id'] ?? null,
                'special' => $validated['special'] ?? false,
                'best_seller' => $validated['best_seller'] ?? false,
                'sort_by' => $validated['sort_by'] ?? 'latest',
            ]
        ], 'Menu berhasil diambil');
    }

    /**
     * Get menu detail
     * GET /api/customer/menu/{id}
     */
    public function show($id): JsonResponse
    {
        // Cache menu detail for 30 minutes
        $menu = Cache::remember("menu_detail_{$id}", 1800, function () use ($id) {
            return Menu::with(['category', 'reviews' => function ($query) {
                $query->where('is_approved', true)
                    ->with('customer:id,name')
                    ->latest()
                    ->limit(5);
            }])
            ->available()
            ->find($id);
        });

        if (!$menu) {
            return $this->errorResponse('Menu tidak ditemukan', 404);
        }

        // Increment view count if needed (for analytics)
        if (!Cache::has("menu_viewed_{$id}_" . request()->ip())) {
            // Here you can add view count logic if needed
            Cache::put("menu_viewed_{$id}_" . request()->ip(), true, 3600);
        }

        return $this->successResponse(
            new MenuResource($menu),
            'Detail menu berhasil diambil'
        );
    }

    /**
     * Get active promos
     * GET /api/customer/promos
     */
    public function getPromos(): JsonResponse
    {
        $promos = Promo::active()
            ->orderBy('end_date', 'asc')
            ->get();

        return $this->successResponse(
            PromoResource::collection($promos),
            'Promo berhasil diambil'
        );
    }

    /**
     * Check promo validity
     * POST /api/customer/promos/check
     */
    public function checkPromo(PromoCheckRequest $request): JsonResponse
    {
        $validated = $request->validated();
        
        $promo = Promo::byCode($validated['code'])->first();
        
        if (!$promo) {
            return $this->errorResponse('Kode promo tidak ditemukan', 404);
        }

        $isValid = $promo->is_active && 
                   $promo->start_date <= now() && 
                   $promo->end_date >= now();
        
        $discountAmount = $promo->calculateDiscount($validated['total_amount']);
        $finalAmount = $validated['total_amount'] - $discountAmount;

        return $this->successResponse([
            'promo' => new PromoResource($promo),
            'is_valid' => $isValid,
            'original_amount' => (float) $validated['total_amount'],
            'discount_amount' => $discountAmount,
            'final_amount' => $finalAmount,
            'validation_message' => $isValid 
                ? 'Promo dapat digunakan' 
                : 'Promo tidak dapat digunakan (tidak aktif atau sudah kadaluarsa)',
            'requirements_met' => !$promo->min_purchase || $validated['total_amount'] >= $promo->min_purchase,
        ], 'Validasi promo berhasil');
    }

    /**
     * Get special menus
     * GET /api/customer/menu/special
     */
    public function getSpecialMenus(): JsonResponse
    {
        $menus = Menu::with('category')
            ->available()
            ->special()
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return $this->successResponse(
            MenuResource::collection($menus),
            'Menu spesial berhasil diambil'
        );
    }

    /**
     * Get best seller menus
     * GET /api/customer/menu/best-seller
     */
    public function getBestSellers(): JsonResponse
    {
        $menus = Menu::with('category')
            ->available()
            ->bestSeller()
            ->orderBy('created_at', 'desc')
            ->limit(10)
            ->get();

        return $this->successResponse(
            MenuResource::collection($menus),
            'Best seller berhasil diambil'
        );
    }
}