<?php
// app/Http/Controllers/Api/Admin/SystemSettingsController.php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Traits\ApiResponse;
use App\Models\SystemSetting;
use App\Models\OperationalHour;
use App\Http\Requests\Api\Admin\SystemSettingRequest;
use App\Http\Resources\Api\SystemSettingResource;
use App\Http\Resources\Api\SystemSettingCollection;
use Illuminate\Http\Request;

class SystemSettingsController extends Controller
{
    use ApiResponse;

    // ========== SYSTEM SETTINGS ==========

    /**
     * Get all system settings with pagination
     */
    public function index(Request $request)
    {
        try {
            $query = SystemSetting::when($request->search, function ($q, $search) {
                    $q->where('key', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%");
                })
                ->when($request->group, function ($q, $group) {
                    $q->where('group', $group);
                })
                ->when($request->type, function ($q, $type) {
                    $q->where('type', $type);
                })
                ->orderBy('group')
                ->orderBy('key');

            $perPage = $request->per_page ?? 50;
            $settings = $query->paginate($perPage);

            return $this->successResponse(
                new SystemSettingCollection($settings),
                'System settings retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve system settings', 500, $e->getMessage());
        }
    }

    /**
     * Get system setting by key
     */
    public function show($key)
    {
        try {
            $setting = SystemSetting::where('key', $key)->first();

            if (!$setting) {
                return $this->errorResponse('System setting not found', 404);
            }

            return $this->successResponse(
                new SystemSettingResource($setting),
                'System setting retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve system setting', 500, $e->getMessage());
        }
    }

    /**
     * Create new system setting
     */
    public function store(SystemSettingRequest $request)
    {
        try {
            // Check for duplicate key
            $existingSetting = SystemSetting::where('key', $request->key)->first();
            if ($existingSetting) {
                return $this->errorResponse('System setting key already exists', 400);
            }

            $setting = SystemSetting::create($request->validated());

            return $this->successResponse(
                new SystemSettingResource($setting),
                'System setting created successfully',
                201
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to create system setting', 500, $e->getMessage());
        }
    }

    /**
     * Update system setting
     */
    public function update(SystemSettingRequest $request, $id)
    {
        try {
            $setting = SystemSetting::find($id);

            if (!$setting) {
                return $this->errorResponse('System setting not found', 404);
            }

            // Check for duplicate key if key is being changed
            if ($request->has('key') && $request->key !== $setting->key) {
                $existingSetting = SystemSetting::where('key', $request->key)->first();
                if ($existingSetting) {
                    return $this->errorResponse('System setting key already exists', 400);
                }
            }

            $setting->update($request->validated());

            return $this->successResponse(
                new SystemSettingResource($setting->fresh()),
                'System setting updated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update system setting', 500, $e->getMessage());
        }
    }

    /**
     * Delete system setting
     */
    public function destroy($id)
    {
        try {
            $setting = SystemSetting::find($id);

            if (!$setting) {
                return $this->errorResponse('System setting not found', 404);
            }

            $setting->delete();

            return $this->successResponse(
                null,
                'System setting deleted successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to delete system setting', 500, $e->getMessage());
        }
    }

    /**
     * Update multiple system settings at once
     */
    public function bulkUpdate(Request $request)
    {
        $request->validate([
            'settings' => 'required|array',
            'settings.*.key' => 'required|exists:system_settings,key',
            'settings.*.value' => 'required'
        ]);

        try {
            foreach ($request->settings as $settingData) {
                $setting = SystemSetting::where('key', $settingData['key'])->first();
                if ($setting) {
                    $setting->update(['value' => $settingData['value']]);
                }
            }

            return $this->successResponse(
                null,
                'System settings updated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update system settings', 500, $e->getMessage());
        }
    }

    // ========== OPERATIONAL HOURS ==========

    /**
     * Get operational hours
     */
    public function getOperationalHours()
    {
        try {
            $hours = OperationalHour::orderBy('day_of_week')->get();

            $daysOfWeek = [
                0 => 'Sunday',
                1 => 'Monday',
                2 => 'Tuesday',
                3 => 'Wednesday',
                4 => 'Thursday',
                5 => 'Friday',
                6 => 'Saturday'
            ];

            $formattedHours = $hours->map(function ($hour) use ($daysOfWeek) {
                return [
                    'id' => $hour->id,
                    'day_of_week' => $hour->day_of_week,
                    'day_name' => $daysOfWeek[$hour->day_of_week] ?? 'Unknown',
                    'is_open' => $hour->is_open,
                    'open_time' => $hour->open_time,
                    'close_time' => $hour->close_time,
                    'formatted_hours' => $hour->is_open 
                        ? "{$hour->open_time} - {$hour->close_time}"
                        : 'Closed'
                ];
            });

            return $this->successResponse(
                $formattedHours,
                'Operational hours retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve operational hours', 500, $e->getMessage());
        }
    }

    /**
     * Update operational hours
     */
    public function updateOperationalHours(Request $request)
    {
        $request->validate([
            'hours' => 'required|array|size:7',
            'hours.*.day_of_week' => 'required|integer|between:0,6',
            'hours.*.is_open' => 'required|boolean',
            'hours.*.open_time' => 'required_if:hours.*.is_open,true|nullable|date_format:H:i',
            'hours.*.close_time' => 'required_if:hours.*.is_open,true|nullable|date_format:H:i|after:hours.*.open_time'
        ]);

        try {
            foreach ($request->hours as $hourData) {
                OperationalHour::updateOrCreate(
                    ['day_of_week' => $hourData['day_of_week']],
                    [
                        'is_open' => $hourData['is_open'],
                        'open_time' => $hourData['is_open'] ? $hourData['open_time'] : null,
                        'close_time' => $hourData['is_open'] ? $hourData['close_time'] : null
                    ]
                );
            }

            return $this->successResponse(
                null,
                'Operational hours updated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update operational hours', 500, $e->getMessage());
        }
    }

    /**
     * Get system dashboard statistics
     */
    public function dashboardStatistics()
    {
        try {
            $totalSettings = SystemSetting::count();
            $publicSettings = SystemSetting::where('is_public', true)->count();
            $byGroup = SystemSetting::selectRaw('group, count(*) as count')
                ->groupBy('group')
                ->get()
                ->pluck('count', 'group');

            $operationalDays = OperationalHour::where('is_open', true)->count();

            return $this->successResponse([
                'total_settings' => $totalSettings,
                'public_settings' => $publicSettings,
                'settings_by_group' => $byGroup,
                'operational_days' => $operationalDays,
                'current_settings' => SystemSetting::whereIn('key', [
                    'app_name',
                    'currency',
                    'tax_rate',
                    'service_fee'
                ])->get()->mapWithKeys(function ($setting) {
                    return [$setting->key => $setting->value];
                })
            ], 'Dashboard statistics retrieved successfully');
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve dashboard statistics', 500, $e->getMessage());
        }
    }
}