<?php
// app/Http\Controllers\Api\Admin\OrderController.php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Traits\ApiResponse;
use App\Models\Order;
use App\Models\OrderItem;
use App\Http\Resources\Api\OrderResource;
use App\Http\Resources\Api\OrderCollection;
use Illuminate\Http\Request;
use Carbon\Carbon;
use PDF;

class OrderController extends Controller
{
    use ApiResponse;

    /**
     * Get all orders with filters
     */
    public function index(Request $request)
    {
        try {
            $query = Order::with(['customer', 'employee', 'payment'])
                ->when($request->search, function ($q, $search) {
                    $q->where('order_number', 'like', "%{$search}%")
                      ->orWhereHas('customer', function ($query) use ($search) {
                          $query->where('name', 'like', "%{$search}%")
                                ->orWhere('email', 'like', "%{$search}%")
                                ->orWhere('phone', 'like', "%{$search}%");
                      });
                })
                ->when($request->status, function ($q, $status) {
                    $q->where('status', $status);
                })
                ->when($request->order_type, function ($q, $orderType) {
                    $q->where('order_type', $orderType);
                })
                ->when($request->date_from, function ($q, $dateFrom) {
                    $q->whereDate('created_at', '>=', $dateFrom);
                })
                ->when($request->date_to, function ($q, $dateTo) {
                    $q->whereDate('created_at', '<=', $dateTo);
                })
                ->orderBy('created_at', 'desc');

            $perPage = $request->per_page ?? 20;
            $orders = $query->paginate($perPage);

            return $this->successResponse(
                new OrderCollection($orders),
                'Orders retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve orders', 500, $e->getMessage());
        }
    }

    /**
     * Get single order
     */
    public function show($id)
    {
        try {
            $order = Order::with([
                'customer',
                'employee',
                'orderItems.menu',
                'payment.paymentMethod'
            ])->find($id);

            if (!$order) {
                return $this->errorResponse('Order not found', 404);
            }

            return $this->successResponse(
                new OrderResource($order),
                'Order retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve order', 500, $e->getMessage());
        }
    }

    /**
     * Update order status
     */
    public function updateStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:pending,confirmed,preparing,ready,completed,cancelled',
            'notes' => 'nullable|string'
        ]);

        try {
            $order = Order::find($id);

            if (!$order) {
                return $this->errorResponse('Order not found', 404);
            }

            $oldStatus = $order->status;
            $order->update([
                'status' => $request->status,
                'notes' => $request->notes
            ]);

            // Jika order selesai, update completed_at
            if ($request->status === 'completed' && !$order->completed_at) {
                $order->update(['completed_at' => Carbon::now()]);
            }

            return $this->successResponse([
                'order' => new OrderResource($order->fresh()->load(['customer', 'orderItems.menu'])),
                'status_changed' => [
                    'from' => $oldStatus,
                    'to' => $request->status
                ]
            ], 'Order status updated successfully');
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update order status', 500, $e->getMessage());
        }
    }

    /**
     * Generate receipt
     */
    public function generateReceipt($id)
    {
        try {
            $order = Order::with([
                'customer',
                'orderItems.menu',
                'payment.paymentMethod'
            ])->find($id);

            if (!$order) {
                return $this->errorResponse('Order not found', 404);
            }

            $receiptData = [
                'order_number' => $order->order_number,
                'customer_name' => $order->customer->name,
                'customer_email' => $order->customer->email,
                'order_type' => $order->order_type,
                'table_number' => $order->table_number,
                'status' => $order->status,
                'order_date' => $order->created_at->format('d/m/Y H:i'),
                'items' => $order->orderItems->map(function ($item) {
                    return [
                        'name' => $item->menu->name,
                        'quantity' => $item->quantity,
                        'price' => $item->unit_price,
                        'subtotal' => $item->subtotal
                    ];
                }),
                'subtotal' => $order->total_amount,
                'tax' => $order->tax_amount,
                'service_fee' => $order->service_fee,
                'discount' => $order->discount_amount,
                'total' => $order->final_amount,
                'payment_method' => $order->payment->paymentMethod->name ?? 'N/A',
                'payment_status' => $order->payment->status ?? 'N/A'
            ];

            // Return JSON receipt (untuk API)
            // Untuk download PDF, bisa dibuat endpoint terpisah
            return $this->successResponse(
                $receiptData,
                'Receipt generated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to generate receipt', 500, $e->getMessage());
        }
    }
}