<?php
// app/Http/Controllers/Api/Admin/MenuController.php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Traits\ApiResponse;
use App\Models\Menu;
use App\Models\Category;
use App\Http\Requests\Api\Admin\MenuRequest;
use App\Http\Resources\Api\MenuResource;
use App\Http\Resources\Api\MenuCollection;
use Illuminate\Http\Request;

class MenuController extends Controller
{
    use ApiResponse;

    /**
     * Get all menus with pagination and search
     */
    public function index(Request $request)
    {
        try {
            $query = Menu::with('category')
                ->when($request->search, function ($q, $search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%");
                })
                ->when($request->category_id, function ($q, $categoryId) {
                    $q->where('category_id', $categoryId);
                })
                ->when($request->is_available !== null, function ($q) use ($request) {
                    $q->where('is_available', $request->is_available);
                })
                ->when($request->is_special !== null, function ($q) use ($request) {
                    $q->where('is_special', $request->is_special);
                })
                ->when($request->is_best_seller !== null, function ($q) use ($request) {
                    $q->where('is_best_seller', $request->is_best_seller);
                })
                ->orderBy('created_at', 'desc');

            $perPage = $request->per_page ?? 20;
            $menus = $query->paginate($perPage);

            return $this->successResponse(
                new MenuCollection($menus),
                'Menus retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve menus', 500, $e->getMessage());
        }
    }

    /**
     * Get single menu
     */
    public function show($id)
    {
        try {
            $menu = Menu::with('category')->find($id);

            if (!$menu) {
                return $this->errorResponse('Menu not found', 404);
            }

            return $this->successResponse(
                new MenuResource($menu),
                'Menu retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve menu', 500, $e->getMessage());
        }
    }

    /**
     * Create new menu
     */
    public function store(MenuRequest $request)
    {
        try {
            // Validate category exists
            $category = Category::find($request->category_id);
            if (!$category) {
                return $this->errorResponse('Category not found', 404);
            }

            $menu = Menu::create($request->validated());

            return $this->successResponse(
                new MenuResource($menu->load('category')),
                'Menu created successfully',
                201
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to create menu', 500, $e->getMessage());
        }
    }

    /**
     * Update menu
     */
    public function update(MenuRequest $request, $id)
    {
        try {
            $menu = Menu::find($id);

            if (!$menu) {
                return $this->errorResponse('Menu not found', 404);
            }

            // Validate category exists if category_id is being updated
            if ($request->has('category_id') && $request->category_id != $menu->category_id) {
                $category = Category::find($request->category_id);
                if (!$category) {
                    return $this->errorResponse('Category not found', 404);
                }
            }

            $menu->update($request->validated());

            return $this->successResponse(
                new MenuResource($menu->fresh()->load('category')),
                'Menu updated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update menu', 500, $e->getMessage());
        }
    }

    /**
     * Delete menu
     */
    public function destroy($id)
    {
        try {
            $menu = Menu::find($id);

            if (!$menu) {
                return $this->errorResponse('Menu not found', 404);
            }

            // Check if menu has order items
            if ($menu->orderItems()->exists()) {
                return $this->errorResponse(
                    'Cannot delete menu because it has associated orders',
                    400
                );
            }

            $menu->delete();

            return $this->successResponse(
                null,
                'Menu deleted successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to delete menu', 500, $e->getMessage());
        }
    }

    /**
     * Bulk update menu status
     */
    public function bulkUpdateStatus(Request $request)
    {
        $request->validate([
            'menu_ids' => 'required|array',
            'menu_ids.*' => 'exists:menus,id',
            'is_available' => 'required|boolean'
        ]);

        try {
            Menu::whereIn('id', $request->menu_ids)
                ->update(['is_available' => $request->is_available]);

            return $this->successResponse(
                null,
                'Menu status updated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update menu status', 500, $e->getMessage());
        }
    }

    /**
     * Get menu statistics
     */
    public function statistics()
    {
        try {
            $totalMenus = Menu::count();
            $availableMenus = Menu::where('is_available', true)->count();
            $specialMenus = Menu::where('is_special', true)->count();
            $bestSellers = Menu::where('is_best_seller', true)->count();
            $outOfStock = Menu::where('stock', 0)->count();

            return $this->successResponse([
                'total_menus' => $totalMenus,
                'available_menus' => $availableMenus,
                'special_menus' => $specialMenus,
                'best_sellers' => $bestSellers,
                'out_of_stock' => $outOfStock,
                'by_category' => Category::withCount(['menus' => function ($query) {
                    $query->where('is_available', true);
                }])->get()
            ], 'Menu statistics retrieved successfully');
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve statistics', 500, $e->getMessage());
        }
    }
}