<?php
// app/Http/Controllers/Api/Admin/CareerController.php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Traits\ApiResponse;
use App\Models\JobPosting;
use App\Models\JobApplication;
use App\Http\Requests\Api\Admin\JobPostingRequest;
use App\Http\Resources\Api\JobPostingResource;
use App\Http\Resources\Api\JobPostingCollection;
use App\Http\Resources\Api\JobApplicationResource;
use App\Http\Resources\Api\JobApplicationCollection;
use Illuminate\Http\Request;
use Carbon\Carbon;

class CareerController extends Controller
{
    use ApiResponse;

    // ========== JOB POSTINGS ==========

    /**
     * Get all job postings with pagination and search
     */
    public function index(Request $request)
    {
        try {
            $query = JobPosting::when($request->search, function ($q, $search) {
                    $q->where('title', 'like', "%{$search}%")
                      ->orWhere('description', 'like', "%{$search}%");
                })
                ->when($request->department, function ($q, $department) {
                    $q->where('department', $department);
                })
                ->when($request->location, function ($q, $location) {
                    $q->where('location', 'like', "%{$location}%");
                })
                ->when($request->employment_type, function ($q, $type) {
                    $q->where('employment_type', $type);
                })
                ->when($request->status, function ($q, $status) {
                    if ($status === 'active') {
                        $q->where('is_active', true)
                          ->where('application_deadline', '>=', now());
                    } elseif ($status === 'expired') {
                        $q->where('application_deadline', '<', now());
                    } elseif ($status === 'inactive') {
                        $q->where('is_active', false);
                    }
                })
                ->orderBy('created_at', 'desc');

            $perPage = $request->per_page ?? 20;
            $jobPostings = $query->paginate($perPage);

            return $this->successResponse(
                new JobPostingCollection($jobPostings),
                'Job postings retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve job postings', 500, $e->getMessage());
        }
    }

    /**
     * Get single job posting with applications
     */
    public function show($id)
    {
        try {
            $jobPosting = JobPosting::with(['applications' => function ($query) {
                $query->orderBy('created_at', 'desc');
            }])->find($id);

            if (!$jobPosting) {
                return $this->errorResponse('Job posting not found', 404);
            }

            return $this->successResponse([
                'job' => new JobPostingResource($jobPosting),
                'applications_count' => $jobPosting->applications->count(),
                'applications_by_status' => $jobPosting->applications->groupBy('status')->map->count()
            ], 'Job posting retrieved successfully');
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve job posting', 500, $e->getMessage());
        }
    }

    /**
     * Create new job posting
     */
    public function store(JobPostingRequest $request)
    {
        try {
            $jobPosting = JobPosting::create($request->validated());

            return $this->successResponse(
                new JobPostingResource($jobPosting),
                'Job posting created successfully',
                201
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to create job posting', 500, $e->getMessage());
        }
    }

    /**
     * Update job posting
     */
    public function update(JobPostingRequest $request, $id)
    {
        try {
            $jobPosting = JobPosting::find($id);

            if (!$jobPosting) {
                return $this->errorResponse('Job posting not found', 404);
            }

            $jobPosting->update($request->validated());

            return $this->successResponse(
                new JobPostingResource($jobPosting->fresh()),
                'Job posting updated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update job posting', 500, $e->getMessage());
        }
    }

    /**
     * Delete job posting
     */
    public function destroy($id)
    {
        try {
            $jobPosting = JobPosting::find($id);

            if (!$jobPosting) {
                return $this->errorResponse('Job posting not found', 404);
            }

            $jobPosting->delete();

            return $this->successResponse(
                null,
                'Job posting deleted successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to delete job posting', 500, $e->getMessage());
        }
    }

    /**
     * Toggle job posting status
     */
    public function toggleStatus($id)
    {
        try {
            $jobPosting = JobPosting::find($id);

            if (!$jobPosting) {
                return $this->errorResponse('Job posting not found', 404);
            }

            $jobPosting->update(['is_active' => !$jobPosting->is_active]);

            return $this->successResponse(
                new JobPostingResource($jobPosting),
                'Job posting status updated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update job posting status', 500, $e->getMessage());
        }
    }

    // ========== JOB APPLICATIONS ==========

    /**
     * Get all job applications
     */
    public function applications(Request $request)
    {
        try {
            $query = JobApplication::with('jobPosting')
                ->when($request->search, function ($q, $search) {
                    $q->where('applicant_name', 'like', "%{$search}%")
                      ->orWhere('applicant_email', 'like', "%{$search}%")
                      ->orWhere('applicant_phone', 'like', "%{$search}%");
                })
                ->when($request->job_posting_id, function ($q, $jobId) {
                    $q->where('job_posting_id', $jobId);
                })
                ->when($request->status, function ($q, $status) {
                    $q->where('status', $status);
                })
                ->when($request->date_from, function ($q, $dateFrom) {
                    $q->whereDate('created_at', '>=', $dateFrom);
                })
                ->when($request->date_to, function ($q, $dateTo) {
                    $q->whereDate('created_at', '<=', $dateTo);
                })
                ->orderBy('created_at', 'desc');

            $perPage = $request->per_page ?? 20;
            $applications = $query->paginate($perPage);

            return $this->successResponse(
                new JobApplicationCollection($applications),
                'Job applications retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve job applications', 500, $e->getMessage());
        }
    }

    /**
     * Get single job application
     */
    public function showApplication($id)
    {
        try {
            $application = JobApplication::with('jobPosting')->find($id);

            if (!$application) {
                return $this->errorResponse('Job application not found', 404);
            }

            return $this->successResponse(
                new JobApplicationResource($application),
                'Job application retrieved successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve job application', 500, $e->getMessage());
        }
    }

    /**
     * Update job application status
     */
    public function updateApplicationStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:pending,reviewed,interviewed,accepted,rejected',
            'notes' => 'nullable|string'
        ]);

        try {
            $application = JobApplication::find($id);

            if (!$application) {
                return $this->errorResponse('Job application not found', 404);
            }

            $application->update([
                'status' => $request->status,
                'notes' => $request->notes
            ]);

            return $this->successResponse(
                new JobApplicationResource($application->fresh()->load('jobPosting')),
                'Job application status updated successfully'
            );
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to update job application', 500, $e->getMessage());
        }
    }

    /**
     * Get career statistics
     */
    public function statistics()
    {
        try {
            $totalJobs = JobPosting::count();
            $activeJobs = JobPosting::where('is_active', true)
                ->where('application_deadline', '>=', now())
                ->count();
            $totalApplications = JobApplication::count();
            $applicationsByStatus = JobApplication::selectRaw('status, count(*) as count')
                ->groupBy('status')
                ->get()
                ->pluck('count', 'status');

            $recentApplications = JobApplication::with('jobPosting')
                ->orderBy('created_at', 'desc')
                ->limit(10)
                ->get();

            $jobsByDepartment = JobPosting::selectRaw('department, count(*) as count')
                ->where('is_active', true)
                ->where('application_deadline', '>=', now())
                ->groupBy('department')
                ->get();

            return $this->successResponse([
                'total_jobs' => $totalJobs,
                'active_jobs' => $activeJobs,
                'total_applications' => $totalApplications,
                'applications_by_status' => $applicationsByStatus,
                'jobs_by_department' => $jobsByDepartment,
                'recent_applications' => $recentApplications
            ], 'Career statistics retrieved successfully');
        } catch (\Exception $e) {
            return $this->errorResponse('Failed to retrieve statistics', 500, $e->getMessage());
        }
    }
}